
/* Copyright (C) 2001-2008 Monotype Imaging Inc. All rights reserved. */

/* Confidential information of Monotype Imaging Inc. */

/* fs_api.c */

#include "fs_itype.h"
#include "fs_ltt.h"

#ifdef FS_PFRR
#include "fs_pfr.h"
#endif

#ifdef FS_EDGE_TECH
#include "adf.h"
#endif

#include "fs_bitmap.h"
#include "fs_graymap.h"

#include "fs_scratch.h"

#ifdef FS_CFFR
#define FS_SCRATCH_SIZE 3000
#else
#define FS_SCRATCH_SIZE 1024
#endif

/*#define FS_GET_ASCENDER_DESCENDER_LEADING__USE_MERGED_TABLES*/

/* some prototypes */
#ifdef FS_DUMP
FS_VOID dump_scaled_fonts(_DS_ FILECHAR *s);
FS_VOID dump_loaded_fonts(_DS_ FILECHAR *s);
#endif

#ifdef FS_STATS
/* gather some statistics to these variables */
FS_ULONG cache_calls;
FS_ULONG cache_probes;
FS_ULONG cached_advance;
FS_ULONG cached_bbox;
FS_ULONG made_outl;
FS_ULONG cached_outl;
FS_ULONG made_bmap;
FS_ULONG cached_bmap;
FS_ULONG made_gmap;
FS_ULONG embedded_bmap;
FS_ULONG cached_gmap;
FS_ULONG made_openvg;
FS_ULONG cached_openvg;
FS_ULONG uncached_too_big;
FS_ULONG collisions;
FS_ULONG num_composite_hints;
FS_ULONG num_composite_chars;
FS_ULONG size_composite_char;
#endif

#ifdef FS_TRACE
int in_char;
int indent_is;
#endif

/* an array used by many folk */
FS_CONST FS_BYTE fs_mask[8] = {0x80, 0x40, 0x20, 0x10, 0x08, 0x04, 0x02, 0x01};

/* some utility functions */

/****************************************************************/
int IDIV(int n, int d)
{
    return (n >= 0) ? n / d : -(1 + (-n - 1) / d);
}

/****************************************************************/
int IMOD(int n, int d)
{
    n = n % d;
    if (n < 0)
        n += d;
    return n;
}

/****************************************************************/
FILECHAR *FS_strdup(_DS_ FILECHAR *s)
{
    int n;
    FILECHAR *r;

    if (s == 0)
        return 0;
    n = SYS_STRLEN(s);
    if (n == 0)
        return 0;
    r = (FILECHAR *)FSS_malloc(_PS_ n + 1);
    if (!r)
        return 0;
    SYS_MEMCPY(r, s, n + 1);
    return r;
}


/****************************************************************/
#ifdef FS_HINTS

/* return a found outline or NULL (not the missing glyph) */
static FS_OUTLINE *x_get_outline(_DS_ FS_LONG unicode)
{
    FS_OUTLINE *outl = 0;
#ifdef FS_RENDER
    FS_ULONG flags = STATE.flags;
    FS_LONG index;
    FS_set_flags(_PS_ FLAGS_CMAP_ON);
    index = LFNT_map_char(_PS_ STATE.cur_lfnt, 0, unicode, 0);
    if (index)
    {
        FS_set_flags(_PS_ FLAGS_CMAP_OFF);

        if (check_sfnt(_PS0_))
            return 0;

        outl = find_or_make_outline(_PS_ STATE.cur_lfnt, STATE.cur_sfnt, unicode, index);
    }
    STATE.flags = flags;
#else
    FS_state_ptr = FS_state_ptr;
    unicode = unicode;
#endif
    return outl;
}

/*****************************************************************************/
/* get some data for rtgah().. */
void get_ref_lines(_DS0_)
{
    F26DOT6 *zones = STATE.cur_sfnt->ref_lines;
    FS_OUTLINE *outl;
    SFNT *sfnt = STATE.cur_sfnt;
    FS_USHORT platform = STATE.platform;
    FS_USHORT encoding = STATE.encoding;
    FS_ULONG flags = STATE.flags;
    FS_ULONG fontflags = STATE.cur_lfnt->fontflags;
    TTF *ttf = (TTF *)STATE.cur_lfnt->fnt;
    FS_ULONG range1 = 0;
    FS_ULONG range2 = 0;
    FS_ULONG range3 = 0;
    TTF_OS2 *pos2 = (TTF_OS2 *)ttf->os2;

    int k;
    fsg_SplineKey *key = 0;
    fnt_GlobalGraphicStateType *gs = 0;
    FS_ULONG b;

    /* only need to do this once per SFNT */
    if (zones[RTGA_BEEN_THERE])
        return;

    SYS_MEMSET(zones, 0, RTGA_COUNT * sizeof(F26DOT6));

    zones[RTGA_BEEN_THERE] = 1;

    /* check panose values for suitability */
    {
        FS_BOOLEAN rtgah_nope = false;
        if (!pos2)
            rtgah_nope = true;
        else
        {
            FS_BYTE panose = pos2->panose[0];
            if ((panose == 3) || (panose == 4) || (panose == 5))
                rtgah_nope = true;
        }
        if (rtgah_nope)
        {
            STATE.cur_sfnt->rtgah_suitable = RTGAH_NOPE;
            FSS_set_cmap(_PS_ platform, encoding);
            STATE.flags = flags;
            return;
        }
    }

    FSS_set_cmap(_PS_ 3, 1);
    if (STATE.error)
    {
        /* try 3,10 instead */
        FSS_set_cmap(_PS_ 3, 10);
        if (STATE.error)
        {
            FSS_set_cmap(_PS_ platform, encoding);
            STATE.flags = flags;
            STATE.cur_lfnt->fontflags = fontflags;
            return;
        }
    }
    FS_set_flags(_PS_ FLAGS_CMAP_ON);
    if (STATE.error)
    {
        FSS_set_cmap(_PS_ platform, encoding);
        STATE.flags = flags;
        STATE.cur_lfnt->fontflags = fontflags;
        return;
    }

    STATE.flags &= ~FLAGS_VERTICAL_ON; /* turn vertical writing flags off */
    STATE.flags &= ~FLAGS_VERTICAL_ROTATE_RIGHT_ON;
    STATE.flags &= ~FLAGS_VERTICAL_ROTATE_LEFT_ON;
    STATE.flags &= ~FLAGS_CHECK_CONTOUR_WINDING_ON; /* turn contour winding check off */

    if (pos2)
    {
        range1 = pos2->ulUnicodeRange1;
        range2 = pos2->ulUnicodeRange2;
        range3 = pos2->ulUnicodeRange3;
    }

    /* ??? no Unicode range set ... assume Latin. */
    if (range1 == 0)
        range1 = 1;

    if (range1 & (1 << 0))  /* Basic Latin */
    {
        /* RTGAH suitability check */
        STATE.flags |= FLAGS_GRAYSCALE;      /* set grayscale    */
        STATE.flags &= ~FLAGS_NO_HINTS;      /* set hints on     */
        STATE.flags |= FLAGS_FORCE_AUTOHINT; /* force RTGAH call */
        STATE.flags |= FLAGS_RTGAH_REF;      /* check for RTGAH suitable */
        outl = x_get_outline(_PS_ 'l');
        STATE.flags &= ~FLAGS_RTGAH_REF;     /* restore the flag */
        /* autohinter may set this flag, we need to preserve it */
        if (outl)
        {
            FSS_free_char(_PS_ outl);
            key = (FS_VOID *)(sfnt->senv->ttkey);
            gs = GLOBALGSTATE(key);
            if (gs->rtgah_data.rtgah_suitable == RTGAH_NOPE)
            {
                FSS_set_cmap(_PS_ platform, encoding);
                STATE.flags = flags;
                STATE.cur_sfnt->rtgah_suitable = RTGAH_NOPE;
                return;
            }
        }

        /* now ... want UNHINTED data, so make sure */
        STATE.flags &= ~FLAGS_GRAYSCALE;      /* no autohinting */
        STATE.flags &= ~FLAGS_FORCE_AUTOHINT; /* no autohinting */
        STATE.flags |= FLAGS_NO_HINTS;        /* no other pre-hinting */

        /* cap_square and base_square look at BEHTZ */
        {
            F26DOT6 hi = MYINFINITY;
            F26DOT6 lo = -MYINFINITY;
            F26DOT6 t;
            FS_CONST char *str = "BEHTZ";

            for (k = 0; k < 5; k++)
            {
                outl = x_get_outline(_PS_ str[k]);
                if (outl)
                {
                    /* want the lowest of the highs */
                    t = outl->hi_y >> 10;
                    if (t < hi) hi = t;

                    /* want the highest of the lows */
                    t = outl->lo_y >> 10;
                    if (t > lo) lo = t;
                    FSS_free_char(_PS_ outl);
                }
            }

            if (hi != MYINFINITY)
                zones[RTGA_CAP_SQUARE] = hi;
            if (lo != -MYINFINITY)
                zones[RTGA_BASE_SQUARE] = lo;
        }

        /* cap round is from "O" */
        outl = x_get_outline(_PS_ 'O');
        if (outl)
        {
            zones[RTGA_CAP_ROUND] = outl->hi_y >> 10;
            zones[RTGA_BASE_ROUND] = outl->lo_y >> 10;
            FSS_free_char(_PS_ outl);
        }

        /* lowercase square looks at w,x,y,z */
        {
            F26DOT6 t, hi = MYINFINITY;
            FS_CONST char *str = "wxyz";

            for (k = 0; k < 4; k++)
            {
                outl = x_get_outline(_PS_ str[k]);
                if (outl)
                {
                    /* want lowest of the highs */
                    t = outl->hi_y >> 10;
                    if (t < hi)
                        hi = t;
                    FSS_free_char(_PS_ outl);
                }
            }
            if (hi != MYINFINITY)
                zones[RTGA_LC_SQUARE] = hi;
        }

        /* lowercase round looks at 'o' */
        outl = x_get_outline(_PS_ 'o');
        if (outl)
        {
            zones[RTGA_LC_ROUND] = outl->hi_y >> 10;
            FSS_free_char(_PS_ outl);
        }
    }
    /* Cyrillic
    when we have as many Cyrillic fonts as Latin, we should probably
    look at multiple characters for these ref_lines too */
    else if (range1 & (1 << 9))
    {
        /* RTGAH suitability check */
        STATE.flags |= FLAGS_GRAYSCALE;      /* set grayscale    */
        STATE.flags &= ~FLAGS_NO_HINTS;      /* set hints on     */
        STATE.flags |= FLAGS_FORCE_AUTOHINT; /* force RTGAH call */
        STATE.flags |= FLAGS_RTGAH_REF;      /* check for RTGAH suitable */
        outl = x_get_outline(_PS_ 0x0418);
        STATE.flags &= ~FLAGS_RTGAH_REF; /* restore the flag */
        /* autohinter may set this flag, we need to preserve it */
        if (outl)
        {
            FSS_free_char(_PS_ outl);
            key = (FS_VOID *)(sfnt->senv->ttkey);
            gs = GLOBALGSTATE(key);
            if (gs->rtgah_data.rtgah_suitable == RTGAH_NOPE)
            {
                FSS_set_cmap(_PS_ platform, encoding);
                STATE.flags = flags;
                STATE.cur_sfnt->rtgah_suitable = RTGAH_NOPE;
                return;
            }
        }

        /* now ... want UNHINTED data, so make sure */
        STATE.flags &= ~FLAGS_GRAYSCALE;      /* no autohinting */
        STATE.flags &= ~FLAGS_FORCE_AUTOHINT; /* no autohinting */
        STATE.flags |= FLAGS_NO_HINTS;        /* no other pre-hinting */

        outl = x_get_outline(_PS_ 0x0415);
        if (outl)
        {
            zones[RTGA_CAP_SQUARE] = outl->hi_y >> 10;
            zones[RTGA_BASE_SQUARE] = outl->lo_y >> 10;
            FSS_free_char(_PS_ outl);
        }

        outl = x_get_outline(_PS_ 0x041E);
        if (outl)
        {
            zones[RTGA_CAP_ROUND] = outl->hi_y >> 10;
            zones[RTGA_BASE_ROUND] = outl->lo_y >> 10;
            FSS_free_char(_PS_ outl);
        }

        outl = x_get_outline(_PS_ 0x043B);
        if (outl)
        {
            zones[RTGA_LC_SQUARE] = outl->hi_y >> 10;
            FSS_free_char(_PS_ outl);
        }

        outl = x_get_outline(_PS_ 0x043E);
        if (outl)
        {
            zones[RTGA_LC_ROUND] = outl->hi_y >> 10;
            FSS_free_char(_PS_ outl);
        }
    }
    else if (range1 & (1 << 7))  /* Greek */
    {
        /* RTGAH suitability check */
        STATE.flags |= FLAGS_GRAYSCALE;      /* set grayscale    */
        STATE.flags &= ~FLAGS_NO_HINTS;      /* set hints on     */
        STATE.flags |= FLAGS_FORCE_AUTOHINT; /* force RTGAH call */
        STATE.flags |= FLAGS_RTGAH_REF;      /* check for RTGAH suitable */
        outl = x_get_outline(_PS_ 0x0399);
        STATE.flags &= ~FLAGS_RTGAH_REF; /* restore the flag */
        /* autohinter may set this flag, we need to preserve it */
        if (outl)
        {
            FSS_free_char(_PS_ outl);
            key = (FS_VOID *)(sfnt->senv->ttkey);
            gs = GLOBALGSTATE(key);
            if (gs->rtgah_data.rtgah_suitable == RTGAH_NOPE)
            {
                FSS_set_cmap(_PS_ platform, encoding);
                STATE.flags = flags;
                STATE.cur_sfnt->rtgah_suitable = RTGAH_NOPE;
                return;
            }
        }

        /* now ... want UNHINTED data, so make sure */
        STATE.flags &= ~FLAGS_GRAYSCALE;      /* no autohinting */
        STATE.flags &= ~FLAGS_FORCE_AUTOHINT; /* no autohinting */
        STATE.flags |= FLAGS_NO_HINTS;        /* no other pre-hinting */

        outl = x_get_outline(_PS_ 0x0395);    /* Epsilon */
        if (outl)
        {
            zones[RTGA_CAP_SQUARE] = outl->hi_y >> 10;
            zones[RTGA_BASE_SQUARE] = outl->lo_y >> 10;
            FSS_free_char(_PS_ outl);
        }

        outl = x_get_outline(_PS_ 0x039F);    /* Omicron */
        if (outl)
        {
            zones[RTGA_CAP_ROUND] = outl->hi_y >> 10;
            zones[RTGA_BASE_ROUND] = outl->lo_y >> 10;
            FSS_free_char(_PS_ outl);
        }

        outl = x_get_outline(_PS_ 0x03C0);    /* pi */
        if (outl)
        {
            zones[RTGA_LC_SQUARE] = outl->hi_y >> 10;
            FSS_free_char(_PS_ outl);
        }

        outl = x_get_outline(_PS_ 0x03BF);    /* omicron */
        if (outl)
        {
            zones[RTGA_LC_ROUND] = outl->hi_y >> 10;
            FSS_free_char(_PS_ outl);
        }
    }

    if (zones[RTGA_l_RSB] == 0)
        /* this can happen only if there is no 'l' reference character in any of
           Latin, Greek, or Cyrillic.  So there will be no latin spacing, so we
           do not a value for the l-RSB.
           BUT ... if it is zero the FUN-FONT detector will assume the next fetched
           character as the 'l' and will (probably) set RTGAH_NOPE.
           So set the l-RSB non-zero -- the value doesn't really matter. */
        zones[RTGA_l_RSB] = 64;


    /* bizarre round/square zones would confuse autohinter
       NOTE: absence of Latin, Cyrillic, Greek reference characters
       will leave the ref_lines all 0.  The following will NOT flag the
       font as bizarre ... for all zero Latin ref_lines. */
    if (zones[RTGA_CAP_ROUND]   < zones[RTGA_CAP_SQUARE]  ||
        zones[RTGA_CAP_SQUARE]  < zones[RTGA_LC_ROUND]    ||
        zones[RTGA_LC_ROUND]    < zones[RTGA_LC_SQUARE]   ||
        zones[RTGA_LC_SQUARE]   < zones[RTGA_BASE_SQUARE] ||
        zones[RTGA_BASE_SQUARE] < zones[RTGA_BASE_ROUND]  )
    {
#ifdef RTGAH_DEBUG
        FS_PRINTF(("ref_lines %d %d %d %d %d %d\n",
                   zones[RTGA_CAP_ROUND], zones[RTGA_CAP_SQUARE], zones[RTGA_LC_ROUND],
                   zones[RTGA_LC_SQUARE], zones[RTGA_BASE_SQUARE], zones[RTGA_LC_ROUND]));
#endif
        STATE.cur_sfnt->rtgah_suitable = RTGAH_NOPE;
        FSS_set_cmap(_PS_ platform, encoding);
        STATE.flags = flags;
        return;
    }

    /* no cap_square and no cap_round ??
     * try getting them from latin figures
     * 1 -- base-square
     * 7 -- cap-square
     * 0 -- both rounds
     *
     * So latin numerals included in Indic fonts will line
     * up with each other (but maybe not with the Indic top)
     */

    if (zones[RTGA_CAP_SQUARE] == 0 && zones[RTGA_CAP_ROUND] == 0)
    {
        /* now ... want UNHINTED data, so make sure */
        STATE.flags &= ~FLAGS_GRAYSCALE;      /* no autohinting */
        STATE.flags &= ~FLAGS_FORCE_AUTOHINT; /* no autohinting */
        STATE.flags |= FLAGS_NO_HINTS;        /* no other pre-hinting */

        outl = x_get_outline(_PS_ '1');
        if (outl)
        {
            zones[RTGA_BASE_SQUARE] = (outl->lo_y) >> 10;
            FSS_free_char(_PS_ outl);
        }
        outl = x_get_outline(_PS_ '7');
        if (outl)
        {
            zones[RTGA_CAP_SQUARE] = (outl->hi_y) >> 10;
            FSS_free_char(_PS_ outl);
        }
        outl = x_get_outline(_PS_ '0');
        if (outl)
        {
            zones[RTGA_CAP_ROUND] = (outl->hi_y) >> 10;
            zones[RTGA_BASE_ROUND] = (outl->lo_y) >> 10;
            FSS_free_char(_PS_ outl);
        }

        /* don't want the x-heights to be 0 ...
           assign to be the same as CAP-heights */
        zones[RTGA_LC_SQUARE] = zones[RTGA_CAP_SQUARE];
        zones[RTGA_LC_ROUND] = zones[RTGA_CAP_ROUND];
    }

    /* 4-2 mods -- for CSong and CSung and other fonts with pothooks */
    if (range2 & (1 << (59 - 32)))
    {
        F26DOT6 ptop = 0;
        F26DOT6 pbot = 0;
        F26DOT6 ftop = 0;
        F26DOT6 fbot = 0;
        int ptop_count = 0;
        int pbot_count = 0;
        int ftop_count = 0;
        int fbot_count = 0;
        F26DOT6 z;

        /* we want these to run through RTGAH to get the stroke info */
        STATE.flags |= FLAGS_GRAYSCALE;      /* set grayscale    */
        STATE.flags &= ~FLAGS_NO_HINTS;      /* set hints on     */
        STATE.flags |= FLAGS_FORCE_AUTOHINT; /* force RTGAH call */

        /* get a hinted single vertical stroke */
        outl = x_get_outline(_PS_ 0x4e28);
        if (outl)
        {
            FSS_free_char(_PS_ outl);
            key = (FS_VOID *)(sfnt->senv->ttkey);
            gs = GLOBALGSTATE(key);
            if (gs->rtgah_data.rtgah_suitable == RTGAH_NOPE)
            {
                FSS_set_cmap(_PS_ platform, encoding);
                STATE.flags = flags;
                STATE.cur_sfnt->rtgah_suitable = RTGAH_NOPE;
                return;
            }
        }

        outl = x_get_outline(_PS_ 0x9748);
        if (outl)
        {
            /* F-F */
            key = (FS_VOID *)(sfnt->senv->ttkey);
            gs = GLOBALGSTATE(key);

            ftop += gs->rtgah_data.hi;  /* stroke positions in F26DOT6 */
            ftop_count++;

            fbot += gs->rtgah_data.lo;
            fbot_count++;

            FSS_free_char(_PS_ outl);
        }

        /* 9748 is not in all our fonts, 9706 is similar */
        outl = x_get_outline(_PS_ 0x9706);
        if (outl)
        {
            /* F-F */
            key = (FS_VOID *)(sfnt->senv->ttkey);
            gs = GLOBALGSTATE(key);

            ftop += gs->rtgah_data.hi;
            ftop_count++;

            fbot += gs->rtgah_data.lo;
            fbot_count++;

            FSS_free_char(_PS_ outl);
        }

        outl = x_get_outline(_PS_ 0x6797);
        if (outl)
        {
            /* P-P */
            ptop += outl->hi_y >> 10;   /* outline coords in 16.16 to 26.6 */
            ptop_count++;

            pbot += outl->lo_y >> 10;
            pbot_count++;

            FSS_free_char(_PS_ outl);
        }

        /* we are set up to use additional controls */
        /* by imitating the previous cases */


        /* use the average of the ref_lines (if any) */
        z = ptop_count ? ptop / ptop_count : MYINFINITY;
        zones[RTGA_CJK_PTOP] = z;

        z = pbot_count ? pbot / pbot_count : MYINFINITY;
        zones[RTGA_CJK_PBOT] = z;

        z = ftop_count ? ftop / ftop_count : MYINFINITY;
        zones[RTGA_CJK_TOP] = z;

        z = fbot_count ? fbot / fbot_count : MYINFINITY;
        zones[RTGA_CJK_BOT] = z;

        /* hints OFF again */
        STATE.flags |= FLAGS_HINTS_OFF;
        STATE.flags &= ~FLAGS_FORCE_AUTOHINT; /* no autohinting */
    }

    /* set flags for all following cases for reflines */
    /* now ... want UNHINTED data, so make sure */
    STATE.flags &= ~FLAGS_GRAYSCALE;      /* no grayscale */
    STATE.flags &= ~FLAGS_FORCE_AUTOHINT; /* no autohinting */
    STATE.flags |= FLAGS_NO_HINTS;        /* no other pre-hinting */

    /* hangul syllables */
    if (range2 & (1 << (56 - 32)))
    {
        outl = x_get_outline(_PS_ 0xD4F0);
        if (outl)
        {
            zones[RTGA_HANGUL_TOP] = outl->hi_y >> 10;
            zones[RTGA_HANGUL_BOT] = outl->lo_y >> 10;
            FSS_free_char(_PS_ outl);
        }
        outl = x_get_outline(_PS_ 0xAC00);
        if (outl)
        {
            zones[RTGA_HANGUL_PTOP] = outl->hi_y >> 10;
            zones[RTGA_HANGUL_PBOT] = outl->lo_y >> 10;
            FSS_free_char(_PS_ outl);
        }
    }

    /* using 93E for Devanagari */
    if (range1 & (1 << 15))
    {
        outl = x_get_outline(_PS_ 0x093e);
        if (outl)
        {
            zones[RTGA_DEVANAGARI_TOP] = outl->hi_y >> 10;
            zones[RTGA_DEVANAGARI_BOT] = outl->lo_y >> 10;
            FSS_free_char(_PS_ outl);
        }
    }

    /* using 995 for Bengali */
    if (range1 & (1 << 16))
    {
        outl = x_get_outline(_PS_ 0x0995);
        if (outl)
        {
            zones[RTGA_BENGALI_TOP] = outl->hi_y >> 10;
            zones[RTGA_BENGALI_BOT] = outl->lo_y >> 10;
            FSS_free_char(_PS_ outl);
        }
    }

    /* using A38 for Gurmukhi */
    if (range1 & (1 << 17))
    {
        outl = x_get_outline(_PS_ 0x0A38);
        if (outl)
        {
            zones[RTGA_GURMUKHI_TOP] = outl->hi_y >> 10;
            zones[RTGA_GURMUKHI_BOT] = outl->lo_y >> 10;
            FSS_free_char(_PS_ outl);
        }
    }

    /* Gujarati */
    if (range1 & (1 << 18))
    {
        outl = x_get_outline(_PS_ 0x0A9F);
        if (outl)
        {
            zones[RTGA_GUJARATI_CAP_ROUND] = outl->hi_y >> 10;
            zones[RTGA_GUJARATI_BASE_ROUND] = outl->lo_y >> 10;
            FSS_free_char(_PS_ outl);
        }

        outl = x_get_outline(_PS_ 0x0ABE);
        if (outl)
        {
            zones[RTGA_GUJARATI_BASE_SQUARE] = 0;
            zones[RTGA_GUJARATI_CAP_SQUARE] = outl->hi_y >> 10;
            FSS_free_char(_PS_ outl);
        }

        /* "but if 0ABE higher than 0A9F ... swap them" */
        /* sounds odd ... but OK                        */
        if (zones[RTGA_GUJARATI_CAP_SQUARE] > zones[RTGA_GUJARATI_CAP_ROUND])
        {
            F26DOT6 z;

            z = zones[RTGA_GUJARATI_CAP_SQUARE];
            zones[RTGA_GUJARATI_CAP_SQUARE] = zones[RTGA_GUJARATI_CAP_ROUND];
            zones[RTGA_GUJARATI_CAP_ROUND] = z;
        }
    }
    /* Tamil */
    if (range1 & (1 << 20))
    {
        outl = x_get_outline(_PS_ 0x0BAA);
        if (outl)
        {
            zones[RTGA_TAMIL_CAP_SQUARE]  = outl->hi_y >> 10;
            zones[RTGA_TAMIL_BASE_SQUARE] = outl->lo_y >> 10;
            FSS_free_char(_PS_ outl);
        }

        outl = x_get_outline(_PS_ 0x0BB1);
        if (outl)
        {
            zones[RTGA_TAMIL_CAP_ROUND] = outl->hi_y >> 10;
            FSS_free_char(_PS_ outl);
        }

        outl = x_get_outline(_PS_ 0x0B95);
        if (outl)
        {
            zones[RTGA_TAMIL_BASE_ROUND] = outl->lo_y >> 10;
            FSS_free_char(_PS_ outl);
        }
    }

    /* Telugu */
    if (range1 & (1 << 21))
    {
        outl = x_get_outline(_PS_ 0x0C05);
        if (outl)
        {
            zones[RTGA_TELUGU_TOP] = outl->hi_y >> 10;
            zones[RTGA_TELUGU_BOT] = outl->lo_y >> 10;
            FSS_free_char(_PS_ outl);
        }
    }

    /* Malayalam */
    if (range1 & (1 << 23))
    {
        outl = x_get_outline(_PS_ 0x0D20);
        if (outl)
        {
            zones[RTGA_MALAYALAM_CAP_ROUND]  = outl->hi_y >> 10;
            zones[RTGA_MALAYALAM_BASE_ROUND] = outl->lo_y >> 10;
            FSS_free_char(_PS_ outl);
        }

        outl = x_get_outline(_PS_ 0x0D1A);
        if (outl)
        {
            zones[RTGA_MALAYALAM_CAP_SQUARE]  = outl->hi_y >> 10;
            zones[RTGA_MALAYALAM_BASE_SQUARE] = outl->lo_y >> 10;
            FSS_free_char(_PS_ outl);
        }
    }

    /* Oriya */
    if (range1 & (1 << 19))
    {
        outl = x_get_outline(_PS_ 0x0B20);
        if (outl)
        {
            zones[RTGA_ORIYA_CAP_ROUND]  = outl->hi_y >> 10;
            zones[RTGA_ORIYA_BASE_ROUND] = outl->lo_y >> 10;
            FSS_free_char(_PS_ outl);
        }
        outl = x_get_outline(_PS_ 0x0B3E);
        if (outl)
        {
            zones[RTGA_ORIYA_BASE_SQUARE] = 0;
            if (outl->lo_y != 0)
                zones[RTGA_ORIYA_CAP_SQUARE] = zones[RTGA_ORIYA_CAP_ROUND];
            else
                zones[RTGA_ORIYA_CAP_SQUARE] = outl->hi_y >> 10;
            FSS_free_char(_PS_ outl);
        }
    }

    /* Sinhala */
    if (range3 & (1 << (73 - 64)))
    {
        outl = x_get_outline(_PS_ 0x0DA7);
        zones[RTGA_SINHALA_CAP_ROUND] = 0;  /* unused */
        zones[RTGA_SINHALA_CAP_SQUARE] = 0; /* unused */
        zones[RTGA_SINHALA_BASE_SQUARE] = 0;

        if (outl)
        {
            zones[RTGA_SINHALA_BASE_ROUND] = outl->lo_y >> 10;
            FSS_free_char(_PS_ outl);
        }
    }

    /* Gujarati -- looks OK without reference line processing */

    /* Thai OE00 - 0E7F */
    if (range1 & (1 << 24))
    {
        outl = x_get_outline(_PS_ 0x0E01);
        if (outl)
        {
            zones[RTGA_THAI_CAP_ROUND] = outl->hi_y >> 10;
            FSS_free_char(_PS_ outl);
        }
        outl = x_get_outline(_PS_ 0x0E40);
        if (outl)
        {
            zones[RTGA_THAI_CAP_SQUARE] = outl->hi_y >> 10;
            FSS_free_char(_PS_ outl);
        }
        outl = x_get_outline(_PS_ 0x0E07);
        if (outl)
        {
            zones[RTGA_THAI_BASE_ROUND] = outl->lo_y >> 10;
            FSS_free_char(_PS_ outl);
        }
        outl = x_get_outline(_PS_ 0x0E1A);
        if (outl)
        {
            zones[RTGA_THAI_BASE_SQUARE] = outl->lo_y >> 10;
            FSS_free_char(_PS_ outl);
        }
    }

    /* Khmer 1780 - 17FF  but NOT Khmer symbols 19E0 - 19FF */
    if (range3 & (1 << (80 - 64)))
    {
        outl = x_get_outline(_PS_ 0x1783);
        if (outl)
        {
            zones[RTGA_KHMER_BASE_SQUARE] = outl->lo_y >> 10;
            FSS_free_char(_PS_ outl);
        }

        outl = x_get_outline(_PS_ 0x1791);
        if (outl)
        {
            zones[RTGA_KHMER_BASE_ROUND] = outl->lo_y >> 10;
            zones[RTGA_KHMER_CAP_ROUND]  = outl->hi_y >> 10;
            FSS_free_char(_PS_ outl);
        }
    }

    /* Arabic */
    b = ((FS_ULONG)1 << (63 - 32));
    if ( (range1 & (1 << 13)) |
            (range2 & b) |
            (range3 & (1 << (67 - 64))))
    {
        outl = x_get_outline(_PS_ 0x062A);
        if (outl)
        {
            zones[RTGAH_ARABIC_BOT] = outl->lo_y >> 10;
            FSS_free_char(_PS_ outl);
        }
    }

    FSS_set_cmap(_PS_ platform, encoding);
    STATE.flags = flags;
    return;

}

#endif /* FS_HINTS */

/****************************************************************/
/* string compare that handles NULL's properly */
int FS_streq(FILECHAR *a, FILECHAR *b)
{
    /* this was the original logic...
    int na = (a==0);
    int nb = (b==0);

    if (na && nb)
        return 1;
    if (na ^ nb)    Coverity had trouble with the bitwise exclusive OR
        return 0;
    return (0==SYS_STRCMP(a,b));
    so this was re-written to eliminate Coverity errors

    the idea is:
        if "a" and "b" are NULL, return "true"
        if only one of "a" or "b" is NULL, return "false"
        otherwise, both non-NULL, do a "strcmp"
    */
    return ((a == 0) ? ((b == 0) ? 1 : 0) : ((b == 0) ? 0 : (SYS_STRCMP(a, b) == 0)));
}


/****************************************************************/
/* quick reality test on the basic typedefs */
static FS_LONG check_types(_DS0_)
{
    FS_TINY c;
    FS_BYTE b;
    FS_SHORT s;
    FS_USHORT us;
    FS_LONG l;
    FS_ULONG ul;

    /* check byte order first */
    {
        /* lower case compiler data types are what's needed here - honest */
        short si;
        char *p;

        si = 1;
        p = (char *)&si;
#if FS_BYTE_ORDER == FS_BIG_ENDIAN
        if (p[0] == 1)
#endif
#if FS_BYTE_ORDER == FS_LITTLE_ENDIAN
            if (p[1] == 1)
#endif
            {
#ifdef FS_DEBUG
                FS_PRINTF(("*** FS_BYTE_ORDER is set incorrectly\n"));
#endif
                return STATE.error = ERR_BYTE_ORDER;
            }
    }

    /*lint -e506  Warning 506: Constant value Boolean */
#ifdef WIN32
    /* disable warning that conditional expression is constant since this is what we want to check */
#pragma warning(disable:4127)
#endif

    c = (FS_TINY)(-1);
    if (1 != sizeof(FS_TINY) || c != -1)
    {
#ifdef FS_DEBUG
        FS_PRINTF(("*** FS_TINY not a signed 8 bit integer\n"));
#endif
        return STATE.error = ERR_TINY_TYPE;
    }

    c = (FS_TINY)0x7F;
    c++;
    if (c > 0)
    {
#ifdef FS_DEBUG
        FS_PRINTF(("*** FS_TINY not a signed 8 bit integer\n"));
#endif
        return STATE.error = ERR_TINY_TYPE;
    }

    c = 0;
    c--;
    if (c > 0)
    {
#ifdef FS_DEBUG
        FS_PRINTF(("*** FS_TINY not a signed 8 bit integer\n"));
#endif
        return STATE.error = ERR_TINY_TYPE;
    }

    b = (FS_BYTE)(-1);
    if (1 != sizeof(FS_BYTE) || b != 0xFF)
    {
#ifdef FS_DEBUG
        FS_PRINTF(("*** FS_BYTE not an unsigned 8 bit integer \n"));
#endif
        return STATE.error = ERR_BYTE_TYPE;
    }

    b = 0xFF;
    b++;
    if (b != 0)
    {
#ifdef FS_DEBUG
        FS_PRINTF(("*** FS_BYTE not an unsigned 8 bit integer \n"));
#endif
        return STATE.error = ERR_BYTE_TYPE;
    }

    s = (FS_SHORT)(-1);
    if (2 != sizeof(FS_SHORT) || s != -1)
    {
#ifdef FS_DEBUG
        FS_PRINTF(("*** FS_SHORT not a signed 16 bit integer\n"));
#endif
        return STATE.error = ERR_SHORT_TYPE;
    }

    s = 0x7FFF;
    s++;
    if (s > 0)
    {
#ifdef FS_DEBUG
        FS_PRINTF(("*** FS_SHORT not a signed 16 bit integer\n"));
#endif
        return STATE.error = ERR_SHORT_TYPE;
    }

    s = 0;
    s--;
    if (s > 0)
    {
#ifdef FS_DEBUG
        FS_PRINTF(("*** FS_SHORT not a signed 16 bit integer\n"));
#endif
        return STATE.error = ERR_SHORT_TYPE;
    }

    us = (FS_USHORT)(-1);
    if (2 != sizeof(FS_USHORT) || us != 0xFFFF)
    {
#ifdef FS_DEBUG
        FS_PRINTF(("*** FS_USHORT not an unsigned 16 bit integer\n"));
#endif
        return STATE.error = ERR_USHORT_TYPE;
    }

    l =  (FS_LONG)(-1);
    if (4 != sizeof(FS_LONG) || l != -1)
    {
#ifdef FS_DEBUG
        FS_PRINTF(("*** FS_LONG not a signed 32 bit integer\n"));
#endif
        return STATE.error = ERR_LONG_TYPE;
    }

    l = 0x7FFFFFFFL;
    l++;
    if (l > 0)
    {
#ifdef FS_DEBUG
        FS_PRINTF(("*** FS_LONG not a signed 32 bit integer\n"));
#endif
        return STATE.error = ERR_LONG_TYPE;
    }

    l = 0;
    l--;
    if (l > 0)
    {
#ifdef FS_DEBUG
        FS_PRINTF(("*** FS_LONG not a signed 32 bit integer\n"));
#endif
        return STATE.error = ERR_LONG_TYPE;
    }


    ul = (FS_ULONG)(-1);
    if (4 != sizeof(FS_ULONG) || ul != 0xFFFFFFFFUL)
    {
#ifdef FS_DEBUG
        FS_PRINTF(("*** FS_ULONG not an unsigned 32 bit integer\n"));
#endif
        return STATE.error = ERR_ULONG_TYPE;
    }


    if (sizeof(long) != SIZEOF_LONG)
    {
#ifdef FS_DEBUG
        FS_PRINTF(("*** SIZEOF_LONG != sizeof(long)\n"));
#endif
        return STATE.error = ERR_SIZEOF_LONG;
    }

    /* if HAS_FS_INT64 is defined ... see if it really works */
#ifdef HAS_FS_INT64
    {
        FS_INT64 aa, bb, cc, dd;

        aa = 12345678;
        bb = 87654321;
        aa += bb;
        cc = aa * bb;
        dd = cc / bb;

        if (dd != aa)
        {
#ifdef FS_DEBUG
            FS_PRINTF(("*** FS_INT64 code doesn't seem to work\n"));
#endif
            return STATE.error = ERR_INT_64;
        }
    }
#endif /* HAS_FS_INT64 */

    /*lint +e506  Warning 506: Constant value Boolean */

    return SUCCESS;
}

static FS_LONG FS_STATE_init(FS_STATE *state)
{
    SYS_MEMSET(state, 0, sizeof(FS_STATE));

    state->platform = state->encoding = 0xFFFF;

    state->varPlatform = 0;
    state->varEncoding = 5;

#ifdef FS_STIK
    state->stroke_pct = FS_DEFAULT_STROKE_PCT;
#endif

    state->outline_width = 1;
    state->outline_opacity = (FS_FIXED)65536; /* 1.0 in 16.16 fixed point */

    return state->error = SUCCESS;
} /*lint !e429 Suppress 'Custodial pointer 'state' has not been freed or returned)' */

#ifdef FS_MEM_DBG
static FS_VOID create_mem_dbg_file(_DS0_)
{
    FILECHAR asciiCount[4];
    FILECHAR dbgfname[20];
    FS_ULONG cntbyte = 2;
    asciiCount[0] = '0';
    asciiCount[1] = '0';
    asciiCount[2] = '0';
    asciiCount[3] =  0 ;
    SYS_STRNCPY(dbgfname, "fs_mem_dbg.txt", 15);
    do
    {
        FILE *fp = SYS_FOPEN(dbgfname, "r");
        if (!fp)
        {
            FS_state_ptr->memdbgfp = SYS_FOPEN(dbgfname, "w");
            if (!FS_state_ptr->memdbgfp)
            {
                FS_FPRINTF((stderr, "PROBLEMS!\n"));
            }
            break;
        }
        else
        {
            FS_ULONG len;

            SYS_FCLOSE(fp);

            if (asciiCount[2] == '9')
            {
                asciiCount[2] = '0';
                if (asciiCount[1] == '9')
                {
                    asciiCount[1] = '0';
                    if (asciiCount[0] == '9')
                    {
                        asciiCount[0] = '0';
                    }
                    else
                    {
                        if (cntbyte > 0) cntbyte = 0;
                        asciiCount[0] += 1;
                    }
                }
                else
                {
                    if (cntbyte > 1) cntbyte = 1;
                    asciiCount[1] += 1;
                }
            }
            else asciiCount[2] += 1;

            SYS_STRNCPY(dbgfname, "fs_mem_dbg_", 12);
            len = SYS_STRLEN(dbgfname);
            SYS_STRNCPY(dbgfname + len, asciiCount + cntbyte, 20 - len);
            len = SYS_STRLEN(dbgfname);
            SYS_STRNCPY(dbgfname + len, ".txt", 20 - len);
        }
    }
    while (!((cntbyte == 0) && !SYS_STRCMP(asciiCount, "099")));
}
#endif /* FS_MEM_DBG */

/****************************************************************/
FS_LONG FSS_init(_DS_ FS_ULONG heap_size)
{
    /* perhaps this should be an exit(STATE.error)... */
    /* that would make sure people notice the error */
    if (check_types(_PS0_))
        return STATE.error;

#ifdef FS_STATS
    /* initialize the statistics */
    cache_calls = 0;
    cache_probes = 0;
    cached_advance = 0;
    cached_bbox = 0;
    made_outl = 0;
    cached_outl = 0;
    made_bmap = 0;
    cached_bmap = 0;
    embedded_bmap = 0;
    made_gmap = 0;
    cached_gmap = 0;
    collisions = 0;
    num_composite_hints = 0;
    num_composite_chars = 0;
    size_composite_char = 0;
    uncached_too_big = 0;
#endif

    /* allocate <server> */
#ifndef FS_MULTI_THREAD
    SYS_MEMSET(&STATE, 0, sizeof(FS_STATE));
#endif    /* !FS_MULTI_THREAD */

    /* allocate <server> */
    STATE.server = (FS_SERVER *) SYS_MALLOC(sizeof(FS_SERVER));
    if (STATE.server == 0)
        return STATE.error = ERR_MALLOC_FAIL;

    SYS_MEMSET(STATE.server, 0, sizeof(FS_SERVER));
    STATE.server->heap_size = heap_size;

    STATE.server->workspaceSize = 0;
    STATE.server->workspace = NULL;
#ifdef FS_GRAYMAPS
    STATE.server->raster = 0;
#endif /* FS_GRAYMAPS */

#ifdef FS_INT_MEM
    STATE.error = __init_heap(_PS_ heap_size, NULL);
    if (STATE.error != SUCCESS)
    {
        SYS_FREE(STATE.server);
        STATE.server = NULL;
        return STATE.error;
    }
#endif

#ifdef FS_MEM_DBG
    create_mem_dbg_file(_PS0_);
#endif

    FsScratchSpace_init(&STATE.server->scratch, FS_state_ptr, FS_SCRATCH_SIZE);

    STATE.parent = NULL;
    STATE.name_map = NULL;

#ifdef FS_STIK
    STATE.stroke_pct = FS_DEFAULT_STROKE_PCT;
#endif

    STATE.server->version = ((FS_ULONG)MAJOR_VERSION << 16) | (FS_USHORT)MINOR_VERSION;
    STATE.platform = STATE.encoding = 0xFFFF;
    STATE.varPlatform = 0;
    STATE.varEncoding = 5;
    STATE.outline_width = 1;
    STATE.outline_opacity = (FS_FIXED)65536; /* 1.0 in 16.16 fixed point */

#ifdef FS_EDGE_TECH
    startADF(_PS0_);
    if (STATE.error)
    {
#ifdef FS_INT_MEM
        __term_heap(_PS0_);
#endif
        SYS_FREE(STATE.server);
        STATE.server = NULL;
        return STATE.error;
    }
#endif

    return STATE.error = SUCCESS;
}

#if defined(FS_INT_MEM) && !defined(FS_MULTI_THREAD)

/****************************************************************/
FS_LONG FS_init_ex(_DS_ FS_VOID *sys_heap, FS_ULONG heap_size)
{
    FS_ULONG usableHeapSize;

    /* the system heap must be passed in to used this function */
    if (sys_heap == NULL)
        return STATE.error = ERR_BAD_EXTERNAL_HEAP;

    /* perhaps this should be an exit(STATE.error)... */
    /* that would make sure people notice the error */
    if (check_types(_PS0_))
        return STATE.error;

#ifdef FS_STATS
    /* initialize the statistics */
    cache_calls = 0;
    cache_probes = 0;
    cached_advance = 0;
    cached_bbox = 0;
    made_outl = 0;
    cached_outl = 0;
    made_bmap = 0;
    cached_bmap = 0;
    embedded_bmap = 0;
    made_gmap = 0;
    cached_gmap = 0;
    collisions = 0;
    num_composite_hints = 0;
    num_composite_chars = 0;
    size_composite_char = 0;
    uncached_too_big = 0;
#endif

    SYS_MEMSET(&STATE, 0, sizeof(FS_STATE));

    /* determine suitable heap size */
    usableHeapSize = (FS_ULONG)heap_size - sizeof(FS_SERVER);
    usableHeapSize += GRANULARITY;
    usableHeapSize &= ~GRANULARITY;
    if (usableHeapSize > (heap_size - sizeof(FS_SERVER)))
    {
        usableHeapSize -= GRANULARITY + 1;
        usableHeapSize &= ~GRANULARITY;
    }


    /* The first sizeof(FS_SERVER) bytes of the user's block will be the state's server */
    /* The remainder will be the usable internal memory heap */

    /* 'allocate' <server> */
    STATE.server = sys_heap;
    if (STATE.server == 0) /* redundant */
        return STATE.error = ERR_MALLOC_FAIL;

    SYS_MEMSET(STATE.server, 0, sizeof(FS_SERVER));
    STATE.server->heap_size = usableHeapSize;

    STATE.server->workspaceSize = 0;
    STATE.server->workspace = NULL;
#ifdef FS_GRAYMAPS
    STATE.server->raster = 0;
#endif /* FS_GRAYMAPS */

    STATE.error = __init_heap(_PS_ usableHeapSize, (FS_BYTE *)sys_heap + sizeof(FS_SERVER));
    if (STATE.error != SUCCESS)
    {
        /*SYS_FREE(STATE.server);*/
        STATE.server = NULL;
        return STATE.error;
    }

#ifdef FS_MEM_DBG
    create_mem_dbg_file(_PS0_);
#endif

    FsScratchSpace_init(&STATE.server->scratch, FS_state_ptr, FS_SCRATCH_SIZE);

    STATE.parent = NULL;
    STATE.name_map = NULL;

#ifdef FS_STIK
    STATE.stroke_pct = FS_DEFAULT_STROKE_PCT;
#endif

    STATE.server->version = ((FS_ULONG)MAJOR_VERSION << 16) | (FS_USHORT)MINOR_VERSION;
    STATE.platform = STATE.encoding = 0xFFFF;
    STATE.varPlatform = 0;
    STATE.varEncoding = 5;
    STATE.outline_width = 1;
    STATE.outline_opacity = (FS_FIXED)65536; /* 1.0 in 16.16 fixed point */

#ifdef FS_EDGE_TECH
    startADF(_PS0_);
    if (STATE.error)
    {
        __term_heap(_PS0_);
        /*SYS_FREE(STATE.server);*/
        STATE.server = NULL;
        return STATE.error;
    }
#endif

    return STATE.error = SUCCESS;
}

#endif  /* FS_INT_MEM and !(FS_MULTI_THREAD) */

/****************************************************************/
FS_LONG FS_error(_DS0_)
{
    return STATE.error;
}

/****************************************************************/
static FS_FIXED
adjustment_scale_factor(FS_FIX88 scale_adjustment, FS_FIXED ppem)
{
    FS_FIXED scale_factor;   /* scale factor = (ppem + adjustment_for_component) / ppem */

    scale_factor = FS_FIX88_TO_FS_FIXED(scale_adjustment);
    scale_factor = FixDiv(ppem + scale_factor, ppem);
    return scale_factor;
}

/****************************************************************/
static FS_LONG
CFNT_get_adjustments(_DS_ FNTSET *fntset, CFNT *cfnt,
                     FS_FIXED yppm, FsPPemAdjust *a)
{
    a->ppem = 0;
    a->scale = 0;
    a->shift = 0;

    if (cfnt->numAdjust)
    {
        FS_BYTE *real_memptr;

        real_memptr = fntset->memptr;

#ifdef FS_MAPPED_FONTS
        /* If "real_memptr" is non-NULL, we assume the font is ROM-based   */
        /* (not memory-mapped) and use "real_memptr" as the address of the */
        /* font. Else, we assume the font is DISK-based and we must make   */
        /* it memory-mapped.                                               */
        if (!real_memptr)
        {
            real_memptr = MF_get_mapped_font(_PS_ fntset->path);
            if (!real_memptr)
            {
                return STATE.error;
            }
        }
#endif
        if (real_memptr)
        {
            FS_USHORT i;
            FILECHAR *buffer = (FILECHAR *)real_memptr + cfnt->adjOffset;

            for (i = 0; i < cfnt->numAdjust; i++)
            {
                FS_USHORT ppem;

                ppem = (buffer[0] << 8) | buffer[1];
                if ((ppem << 16) > yppm)
                {
                    break;
                }
                else
                {
                    a->ppem = ppem;
                    a->scale = (buffer[2] << 8) | buffer[3];
                    a->shift = (buffer[4] << 8) | buffer[5];

                    if ((ppem << 16) == yppm) break;
                }
                buffer += sizeof(FsPPemAdjust);
            }
        }
        else
        {
            FS_FILE *fp;

            fp = FS_open(_PS_ fntset->path);
            if (fp)
            {
                FS_USHORT i;

                FS_seek(_PS_ fp, cfnt->adjOffset, SEEK_SET);

                for (i = 0; i < cfnt->numAdjust; i++)
                {
                    FILECHAR buffer[sizeof(FsPPemAdjust)];
                    FS_USHORT ppem;

                    FS_read(_PS_ fp, (FS_BYTE *)buffer, sizeof(FsPPemAdjust));
                    if (STATE.error)
                    {
                        FS_close(_PS_ fp);
                        return STATE.error;
                    }
                    ppem = (buffer[0] << 8) | buffer[1];
                    if ((ppem << 16) > yppm)
                    {
                        break;
                    }
                    else
                    {
                        a->ppem = ppem;
                        a->scale = (buffer[2] << 8) | buffer[3];
                        a->shift = (buffer[4] << 8) | buffer[5];

                        if ((ppem << 16) == yppm) break;
                    }
                }
                FS_close(_PS_ fp);
            }
            else
            {
                return STATE.error;
            }
        }
    }
    return STATE.error = SUCCESS;
}

/****************************************************************/

/**
* Set the scale transformation matrix.
* args are in 16.16 PIXELS ... (12<<16),0,0,(12<<16) for 12 pixels_per_em
* Does not load the scaled font at this point. See check_sfnt().
*/
FS_LONG FSS_set_scale(_DS_ FS_FIXED s00, FS_FIXED s01, FS_FIXED s10, FS_FIXED s11)
{
    TFNT *tfnt;
    FS_FIXED xppm, yppm, tan_s;
    FS_LONG err;
    FS_USHORT component;
    int stik = 0;

    /* gotta be a current font selection */
    if (STATE.cur_typeset.fntset == 0)
        return STATE.error = ERR_NO_CURRENT_FNTSET;

    /* store values specified by the user */
    STATE.scale[0] = s00;
    STATE.scale[1] = s01;
    STATE.scale[2] = s10;
    STATE.scale[3] = s11;

    err = get_scale_inputs(STATE.scale, &xppm, &yppm, &tan_s);
    if (err)
        return STATE.error = err;

    STATE.lpm = FS_ROUND(yppm); /* rotationally invariant measure of size */

    /* find or set essential scale information for each component font */
    for (component = 0, tfnt = STATE.cur_typeset.tfntarray;
         component < STATE.cur_typeset.fntset->num_fonts;
         component++, tfnt++)
    {
        CFNT *cfnt = 0;
        SFNT *sfnt, *prev;
        LFNT *lfnt;
        FS_FIXED scale[4];       /* holds the adjusted scale factors for component */
        FS_SHORT vertical_shift; /* shift adjustment for component */

        if (tfnt)
            cfnt = tfnt->cfnt;
        else
            return STATE.error = ERR_BAD_TYPESET;

        lfnt = (LFNT *)(cfnt->lfnt);

        /* if this is the metric font, name as current font */
        if (STATE.cur_typeset.fntset->metric_font == lfnt)
            STATE.cur_font = component;

#if defined(FS_STIK)
        stik = (lfnt->fontflags & FONTFLAG_STIK) == FONTFLAG_STIK;
#endif
        /* scale prior to adjustments */
        scale[0] = s00;
        scale[1] = s01;
        scale[2] = s10;
        scale[3] = s11;

        /* Apply linked font scale adjustments */
        {
            FsPPemAdjust adjustments;

            CFNT_get_adjustments(_PS_ STATE.cur_typeset.fntset, cfnt, yppm, &adjustments);
            if (STATE.error)
                return STATE.error;

            if (adjustments.scale)
            {
                FS_FIXED scale_factor;   /* scale factor = (ppem + adjustment_for_component) / ppem */

                scale_factor = adjustment_scale_factor(adjustments.scale, yppm);

                scale[0] = FixMul(scale[0], scale_factor);
                scale[1] = FixMul(scale[1], scale_factor);
                scale[2] = FixMul(scale[2], scale_factor);
                scale[3] = FixMul(scale[3], scale_factor);
            }

            vertical_shift = adjustments.shift;
        }
#if defined(FS_STIK_EVENS_ONLY)
        /* Adjust stik font scale for EVENS ONLY if defined      */
        /* for ESQ Mobile fonts with SmartHint technology        */
        /* and vanilla scale transformations,                    */
        /* if odd size specified, change to next lower even size */
        if ( (lfnt->fontflags & FONTFLAG_NEW_AA_ON) &&
             (scale[0] > 0 && scale[1] == 0 && scale[2] == 0 && scale[3] > 0) )
        {
            scale[0] = scale[0] >> 16;
            scale[3] = scale[3] >> 16;
            if (scale[0] & 1) scale[0] -= 1;
            if (scale[3] & 1) scale[3] -= 1;
            scale[0] = scale[0] << 16;
            scale[3] = scale[3] << 16;
        }
#endif

        /* apply component font italic angle */
        if (cfnt->italicize)
        {
            STATE.error = modify_scale(scale, cfnt->italic_angle);
            if (STATE.error)
                return STATE.error;
        }

        /* cruise existing scaled fonts for one with proper ttf and adjusted scale */
        STATE.error = SUCCESS;
        for (sfnt = STATE.server->scaled_fonts, prev = NULL;
             sfnt;
             prev = sfnt, sfnt = (SFNT *)(sfnt->next))
        {
            FS_FIXED *us;

            if ((LFNT *)(sfnt->lfnt) != lfnt)
                continue;

            us = sfnt->user_scale;

            if (us[0] == scale[0] && us[1] == scale[1] && us[2] == scale[2] && us[3] == scale[3]

                    /* must match vertical shift for specified size */
                    && (sfnt->vertical_shift == vertical_shift)

                    /* if a STIK font THEN the stroke width must match */
                    /* I'm sure we all recall from logic class that */
                    /* A IMPLIES B is equivalent to: NOT(A) OR B */
                    && (!stik || sfnt->stroke_pct == STATE.stroke_pct)

#ifdef FS_PSEUDO_BOLD
                    /* the bold percents must match too */
                    && (sfnt->bold_pct == (STATE.bold_pct + cfnt->bold_pct))
#endif
               ) /* ")" to end the "if (us[0]== ..." clause */
            {
                /* found a matching sfnt                              */
                /* move <sfnt> to front of <SFNTs> ... LRU management */
                if (sfnt != (SFNT *)(STATE.server->scaled_fonts))
                {
                    /* unlink */
                    if (prev)
                        prev->next = sfnt->next;

                    /* prepend */
                    sfnt->next = STATE.server->scaled_fonts;
                    STATE.server->scaled_fonts = sfnt;
                }

                break; /* sfnt found */
            } /* if matching sfnt found */
        } /* for each sfnt in server linked list */

        if (!sfnt)
        {
            /* not found ... make a new scaled font entry */
#ifdef FS_MEM_DBG
            STATE.memdbgid = "SFNT";
#endif
            sfnt = (SFNT *)FSS_calloc(_PS_ sizeof(SFNT));
            if (sfnt == NULL) /* Coverity did not like "if (STATE.error)" */
                return STATE.error;

            sfnt->lfnt = lfnt;
            sfnt->vertical_shift = vertical_shift;
            SYS_MEMCPY(sfnt->user_scale, scale, 16);
            sfnt->stroke_pct = STATE.stroke_pct; /* save in sfnt for caching purposes */
            sfnt->bold_pct = STATE.bold_pct + cfnt->bold_pct;
#ifdef FS_HINTS
            sfnt->rtgah_suitable = RTGAH_YES;  /* unless it is found to be unsuitable later */
#endif

            /* defer scaling the font until actually needed */

            /* prepend to list of scaled fonts */
            sfnt->next = STATE.server->scaled_fonts;
            STATE.server->scaled_fonts = sfnt;
        }

        /* name as current SFNT */
        if (tfnt->sfnt) tfnt->sfnt->ref_count--;
        tfnt->sfnt = sfnt;
        sfnt->ref_count++;

#ifdef FS_EDGE_RENDER
        /* reset the current Edge adjustments */
        FS_set_csm_adjustments(_PS_ STATE.sharpnessOffset, STATE.sharpnessSlope,
                               STATE.thicknessOffset, STATE.thicknessSlope);

        /* set up Edge scaling */
        adfComputeTypographicScaling(_PS_ sfnt, &tfnt->adfRenderAttrs);
        if (STATE.error)
            return STATE.error;
#endif

        STATE.cur_sfnt = sfnt;
        STATE.cur_lfnt = lfnt;
        STATE.cur_font = component;
        /* defer getting RTGAH ref lines until actually needed */

    } /* for each component font */

    return STATE.error = SUCCESS;
}

/**
* Get the scale transformation matrix.
* args are in 16.16 PIXELS ... (12<<16),0,0,(12<<16) for 12 pixels_per_em
*/
FS_LONG FSS_get_scale(_DS_ FS_FIXED *s00, FS_FIXED *s01, FS_FIXED *s10, FS_FIXED *s11)
{
    SFNT *sfnt;

    /* check whether font selected */
    if (STATE.cur_typeset.fntset == 0)
        return STATE.error = ERR_NO_CURRENT_FNTSET;

    /* check whether scale has been set */
    if (STATE.cur_typeset.tfntarray[STATE.cur_font].sfnt == 0)
        return STATE.error = ERR_NO_CURRENT_SFNT;

    sfnt = STATE.cur_typeset.tfntarray[STATE.cur_font].sfnt;
    *s00 = sfnt->user_scale[0];
    *s01 = sfnt->user_scale[1];
    *s10 = sfnt->user_scale[2];
    *s11 = sfnt->user_scale[3];
    return STATE.error = SUCCESS;
}

/**
*  Get the font design units per em associated with the current
*  component font that was last used to render a glyph
*/
FS_USHORT FSS_get_design_units(_DS0_)
{
    TFNT *tfnt;
    CFNT *cfnt = 0;
    LFNT *lfnt;
    TTF  *ttf;
    FS_USHORT design_units;
    FNTSET *fntset = STATE.cur_typeset.fntset;

    /* check whether font selected */
    if (fntset == 0)
    {
        STATE.error = ERR_NO_CURRENT_FNTSET;
        return 0;
    }

    tfnt = &STATE.cur_typeset.tfntarray[STATE.cur_font];
    if (tfnt)
    {
        cfnt = (CFNT *)(tfnt->cfnt);
        if (cfnt)
            lfnt = (LFNT *)(cfnt->lfnt);
        else
            lfnt = (LFNT *)(fntset->metric_font);
    }
    else
        lfnt = (LFNT *)(fntset->metric_font);

    ttf = (TTF *)(lfnt->fnt);
    if (!ttf)
    {
        STATE.error = ERR_BAD_LFNT;
        return 0;
    }

    design_units = ttf->head->unitsPerEm;
    return design_units;
}

/**
* Set platform/encoding/current cmap in STATE as requested.
*/
FS_LONG FSS_set_cmap(_DS_ FS_USHORT platform, FS_USHORT encoding)
{
    /* clear the current cmap for each component of the current set */
    TFNT *tfnt = STATE.cur_typeset.tfntarray;
    FS_USHORT n;

    if(platform == STATE.platform && encoding == STATE.encoding)
        return STATE.error = SUCCESS;

    for (n=0; n<STATE.cur_typeset.capacity; n++)
    {
        tfnt->cmap_offset = 0;
        tfnt++;
    }

    /* OK, set the cmap, platform, encoding for first component */
    STATE.error = SUCCESS;
    map_font(_PS_ STATE.cur_typeset.tfntarray, platform, encoding);

    STATE.platform = platform;
    STATE.encoding = encoding;

    return STATE.error;
}

/**
* Get the current platform/encoding in STATE.
*/
FS_LONG FSS_get_cmap(_DS_ FS_USHORT *platform, FS_USHORT *encoding)
{
    *platform = STATE.platform;
    *encoding = STATE.encoding;
    return STATE.error = SUCCESS;
}

/**
* Set variant platform/encoding in STATE as requested.
*/
FS_LONG FS_set_cmap_variant(_DS_ FS_USHORT varPlatform, FS_USHORT varEncoding)
{
    STATE.varPlatform = varPlatform;
    STATE.varEncoding = varEncoding;
    return STATE.error = SUCCESS;
}

/**
* Get the current variant platform/encoding in STATE.
*/
FS_LONG FS_get_cmap_variant(_DS_ FS_USHORT *variantPlatform, FS_USHORT *variantEncoding)
{
    *variantPlatform = STATE.varPlatform;
    *variantEncoding = STATE.varEncoding;
    return STATE.error = SUCCESS;
}

/**
* Get the current flags setting in STATE.
*/
FS_LONG FS_get_flags(_DS0_)
{
    return STATE.flags;
}

/**
* Set an effect flag in STATE.
* Can only set/clear one effect at a time.
*/
FS_LONG FS_set_flags(_DS_ FS_ULONG effect)
{
    STATE.error = SUCCESS;

    /* most common misc effects */
    if (effect == FLAGS_CMAP_OFF || effect == FLAGS_DROPOUTS_ON || effect == FLAGS_HINTS_OFF)
    {
        STATE.flags |= effect;
    }
    else if (effect == FLAGS_CMAP_ON || effect == FLAGS_DROPOUTS_OFF || effect == FLAGS_HINTS_ON)
    {
        STATE.flags &= effect;
    }

    /* mutually exclusive bitmap effects */
    else if (effect == FLAGS_EMBOSSED || effect == FLAGS_ENGRAVED ||
             effect == FLAGS_OUTLINED_1PIXEL || effect == FLAGS_OUTLINED_2PIXEL ||
             effect == FLAGS_OUTLINED_UNFILLED || effect == FLAGS_OUTLINED_FILLED ||
             effect == FLAGS_OUTLINED_SOFT)
    {
        /* mutually exclusive ... clear all, set one */
        STATE.flags &= FLAGS_NO_EFFECT;
        STATE.flags |= effect;
    }
    else if ( effect == FLAGS_ADD_WEIGHT )
    {
#ifdef FS_PSEUDO_BOLD
        /* mutually exclusive ... clear all, set one */
        STATE.flags &= FLAGS_REGULARBOLD_OFF;
        STATE.flags &= FLAGS_FRACTBOLD_OFF;
        STATE.flags |= effect;
#else
        STATE.error = ERR_BOLD_UNDEF;
#endif
    }
    else if (effect == FLAGS_NO_EFFECT)
    {
        STATE.flags &= effect;
    }

    /* emboldening flags */
    else if ( effect == FLAGS_REGULARBOLD )
    {
#ifdef FS_PSEUDO_BOLD
        /* mutually exclusive ... clear others, set the one */
        STATE.flags &= FLAGS_ADD_WEIGHT_OFF;
        STATE.flags &= FLAGS_FRACTBOLD_OFF;
        STATE.flags |= effect;
#else
        STATE.error = ERR_BOLD_UNDEF;
#endif
    }
    else if ( effect == FLAGS_FRACTBOLD )
    {
#ifdef FS_PSEUDO_BOLD
        /* mutually exclusive ... clear others, set the one */
        STATE.flags &= FLAGS_ADD_WEIGHT_OFF;
        STATE.flags &= FLAGS_REGULARBOLD_OFF;
        STATE.flags |= effect;
#else
        STATE.error = ERR_BOLD_UNDEF;
#endif
    }
    else if ( effect == FLAGS_REGULARBOLD_OFF)
    {
#ifdef FS_PSEUDO_BOLD
        STATE.flags &= effect;
#else
        STATE.error = ERR_BOLD_UNDEF;
#endif
    }
    else if ( effect == FLAGS_ADD_WEIGHT_OFF)
    {
#ifdef FS_PSEUDO_BOLD
        STATE.flags &= effect;
#else
        STATE.error = ERR_BOLD_UNDEF;
#endif
    }
    else if ( effect == FLAGS_FRACTBOLD_OFF)
    {
#ifdef FS_PSEUDO_BOLD
        STATE.flags &= effect;
#else
        STATE.error = ERR_BOLD_UNDEF;
#endif
    }
    else if (effect == FLAGS_SOFTENED_ON)
    {
        STATE.flags |= effect;
    }
    else if (effect == FLAGS_SOFTENED_OFF ||
             effect == FLAGS_NO_GRAYMAP_EFFECT)
    {
        STATE.flags &= effect;
    }
    else if (effect == FLAGS_VERTICAL_ON)
    {
        /* mutually exclusive ... clear others, set the one */
        STATE.flags &= FLAGS_VERTICAL_ROTATE_RIGHT_OFF;
        STATE.flags &= FLAGS_VERTICAL_ROTATE_LEFT_OFF;
        STATE.flags |= effect;
    }
    else if (effect == FLAGS_VERTICAL_OFF)
    {
        /* mutually exclusive ... so clear them all just in case */
        STATE.flags &= FLAGS_VERTICAL_OFF;
        STATE.flags &= FLAGS_VERTICAL_ROTATE_RIGHT_OFF;
        STATE.flags &= FLAGS_VERTICAL_ROTATE_LEFT_OFF;
    }
    else if (effect == FLAGS_VERTICAL_ROTATE_RIGHT_ON)
    {
        /* mutually exclusive ... clear others, set the one */
        STATE.flags &= FLAGS_VERTICAL_OFF;
        STATE.flags &= FLAGS_VERTICAL_ROTATE_LEFT_OFF;
        STATE.flags |= effect;
    }
    else if (effect == FLAGS_VERTICAL_ROTATE_RIGHT_OFF)
    {
        /* mutually exclusive ... so clear them all just in case */
        STATE.flags &= FLAGS_VERTICAL_OFF;
        STATE.flags &= FLAGS_VERTICAL_ROTATE_RIGHT_OFF;
        STATE.flags &= FLAGS_VERTICAL_ROTATE_LEFT_OFF;
    }
    else if (effect == FLAGS_VERTICAL_ROTATE_LEFT_ON)
    {
        /* mutually exclusive ... clear others, set the one */
        STATE.flags &= FLAGS_VERTICAL_OFF;
        STATE.flags &= FLAGS_VERTICAL_ROTATE_RIGHT_OFF;
        STATE.flags |= effect;
    }
    else if (effect == FLAGS_VERTICAL_ROTATE_LEFT_OFF)
    {
        /* mutually exclusive ... so clear them all just in case */
        STATE.flags &= FLAGS_VERTICAL_OFF;
        STATE.flags &= FLAGS_VERTICAL_ROTATE_RIGHT_OFF;
        STATE.flags &= FLAGS_VERTICAL_ROTATE_LEFT_OFF;
    }
    else if (effect == FLAGS_GRAYSCALE_ON)
    {
        STATE.flags |= effect;
    }
    else if (effect == FLAGS_GRAYSCALE_OFF)
    {
        STATE.flags &= effect;
    }
    /* pseudo bold contour checking */
    else if (effect == FLAGS_CHECK_CONTOUR_WINDING_ON)
    {
#ifdef FS_CONTOUR_WINDING_DETECTION
        STATE.flags |= effect;
#else
        STATE.error = ERR_CONTOURCHECK_UNDEF;
#endif
    }
    else if (effect == FLAGS_CHECK_CONTOUR_WINDING_OFF)
    {
#ifdef FS_CONTOUR_WINDING_DETECTION
        STATE.flags &= effect;
#else
        STATE.error = ERR_CONTOURCHECK_UNDEF;
#endif
    }
#ifdef FS_EDGE_RENDER
    else if (effect == FLAGS_DEFAULT_CSM_ON)
    {
        STATE.flags |= effect;
    }
    else if (effect == FLAGS_DEFAULT_CSM_OFF)
    {
        STATE.flags &= effect;
    }
#endif
#ifdef FS_EDGE_HINTS
    else if (effect == FLAGS_MAZ_ON)
    {
        STATE.flags |= effect;
    }
    else if (effect == FLAGS_MAZ_OFF)
    {
        STATE.flags &= effect;
    }
#endif
#ifdef FS_HINTS
    else if (effect == FLAGS_AUTOHINT_OFF)
    {
        STATE.flags |= effect;
    }
    else if (effect == FLAGS_AUTOHINT_ON)
    {
        STATE.flags &= effect;
    }
    else if (effect == FLAGS_FORCE_AUTOHINT)
    {
        STATE.flags |= effect;
    }
    else if (effect == FLAGS_FORCE_AUTOHINT_OFF)
    {
        STATE.flags &= effect;
    }
    else if (effect == FLAGS_AUTOHINT_YONLY_ON)
    {
        STATE.flags |= effect;
    }
    else if (effect == FLAGS_AUTOHINT_YONLY_OFF)
    {
        STATE.flags &= effect;
    }
#endif
    else if (effect == FLAGS_KERNING_SCALE_OFF)
    {
        STATE.flags |= effect;
    }
    else if (effect == FLAGS_KERNING_SCALE_ON)
    {
        STATE.flags &= effect;
    }
    /* bad argument */
    else
        STATE.error = ERR_BAD_EFFECT;
    return STATE.error;
}

/****************************************************************/
FS_LONG FS_set_bold_pct(_DS_ FS_FIXED pct)
{
#ifdef FS_PSEUDO_BOLD
    if (pct < 0 || pct > 65536L)  /* 0 ... 100 percent */
        return STATE.error = ERR_BAD_PERCENT;
    STATE.bold_pct = pct;
    return STATE.error = SUCCESS;
#else
    pct = pct;
    return STATE.error = ERR_BOLD_UNDEF;
#endif
}

/****************************************************************/
FS_LONG FS_set_stroke_pct(_DS_ FS_FIXED pct)
{
#ifdef FS_STIK
    if (pct < 0 || pct > 32768)  /* 0 ... 50 percent */
        return STATE.error = ERR_BAD_PERCENT;
    STATE.stroke_pct = pct;
    return STATE.error = SUCCESS;
#else
    pct = pct;
    return STATE.error = ERR_STIK_UNDEF;
#endif
}

/****************************************************************/
FS_LONG FS_set_outline_width(_DS_ FS_USHORT width)
{
    if ( width == 0 || width > 10 )
        return STATE.error = ERR_BAD_OUTLINE_WIDTH;

    STATE.outline_width = width;
    return STATE.error = SUCCESS;
}
/****************************************************************/
FS_LONG FS_set_outline_opacity(_DS_ FS_FIXED opacity)
{
    if ( opacity < 0 || opacity > 655360L)
        return STATE.error = ERR_BAD_OUTLINE_OPACITY;

    STATE.outline_opacity = opacity;
    return STATE.error = SUCCESS;
}

#ifdef FS_RENDER

/**
 * determine which outline to return and return
 * either the outline in the STATE or one constructed from
 * the font data
 */
FS_OUTLINE *internal_user_get_outline(_DS_ FS_ULONG id)
{
#ifdef FS_EXTERNAL_OUTLINE
    if (STATE.user_outline)
    {
        return copy_outline(_PS_ STATE.user_outline,
                            0 /* source is in server memory */);
    }
    else
#endif
    {
        FS_OUTLINE *outl = internal_get_outline(_PS_ id);

#ifdef WANT_TO_FORCE_STIK_TO_SAUSAGE
#ifdef FS_STIK
        if (outl && (outl->num > 0) &&
            (STATE.cur_lfnt->fontflags & FONTFLAG_STIK) &&
            !(outl->type[0] & OUTLINE_CHAR))
        {
            FS_LONG npoints, ntypes;
            FS_OUTLINE *expo = expand_stik(_PS_ outl, &ntypes, &npoints);
            FSS_free_char(_PS_ outl);
            outl = expo;
        }
#endif
#endif
        return outl;
    }
}
#endif /* FS_RENDER */

/**
 * retrieve a scaled and possibly hinted outline from either
 * the STATE or the font
 */
FS_OUTLINE *FSS_get_outline(_DS_ FS_ULONG id)
{
    FS_OUTLINE *outl;

#ifdef FS_RENDER
    outl = internal_user_get_outline(_PS_ id);

#else
    FS_state_ptr = FS_state_ptr;
    id = id;
    outl = 0;
#endif /* FS_RENDER */
    return outl;
}

/**
 * get a bitmap by either retrieving it from cache,
 * retrieving an embedded bitmap, or rendering it from an outline
 */
FS_BITMAP *FSS_get_bitmap(_DS_ FS_ULONG id)
{
    FS_BITMAP *bmap;
#if defined(FS_BITMAPS) || defined(FS_EMBEDDED_BITMAP)
    FS_USHORT index; /* index ignored for user outline */

#ifdef FS_EXTERNAL_OUTLINE
    index = 0;
    if (!STATE.user_outline)
#endif
    {
        /* character id to index */
        index = map_char(_PS_ id, 0);
        if (STATE.error)
            return 0;

        if (check_sfnt(_PS0_))
            return 0;
        /* no need to get RTGAH ref lines for bitmaps */
    }

    bmap = get_bitmap(_PS_  id, index);
    if (STATE.error)
        return 0;
#else
    bmap = 0;
#endif
    return bmap;
}

/**
 * get a graymap by either retrieving it from cache,
 * retrieving an embedded graymap, or rendering it from an outline
 */
FS_GRAYMAP *FSS_get_graymap(_DS_ FS_ULONG id)
{
    FS_GRAYMAP *gmap;
#if defined(FS_GRAYMAPS) || defined(FS_EMBEDDED_BITMAP)
    FS_USHORT index; /* index ignored for user outline */

#ifdef FS_EXTERNAL_OUTLINE
    index = 0;
    if (!STATE.user_outline)
#endif
    {
        /* character id to index */
        index = map_char(_PS_ id, 0);
        if (STATE.error)
            return 0;

        if (check_sfnt(_PS0_))
            return 0;
#ifdef FS_HINTS
        if ((STATE.cur_lfnt->fontflags & FONTFLAG_STIK) != FONTFLAG_STIK &&
            (STATE.cur_lfnt->fnt_type != PFR_TYPE) &&
            (STATE.cur_sfnt->ref_lines[RTGA_BEEN_THERE] == 0))
            get_ref_lines(_PS0_); /* get RTGAH ref lines since not already done */
#endif
    }

    gmap = get_graymap(_PS_  id, index, FS_MAP_GRAYMAP4,0,0);
    if (STATE.error)
        return 0;
#else
    FS_state_ptr = FS_state_ptr;
    id = id;
    gmap = 0;
#endif
    return gmap;
}

/**
 * resolve the glyphmap type to produce
 * depending on the type requested.
 * The precedence hierarchy is:
 * raster icon, vector icon, ADF graymap, graymap (4, 8, or 2-bit), bitmap
 * otherwise:
 * raster icon, vector icon, ADF graymap(if adfh table exists), graymap (4, 8, or 2-bit),
 * bitmap when gasp does not exist or GASP_GRIDFIT bit is on in gasp table, otherwise,
 * raster icon, vector icon, bitmap, ADF graymap(if adfh table exists),
 * graymap (4, 8, or 2-bit),
 */
static FS_VOID resolve_glyphmap_type(_DS_ FS_USHORT *type, FS_USHORT *rasterizer)
{
    FS_USHORT type_requested = 0;
    SENV* senv;
    FS_USHORT rbv=0;

    if (*type & FS_MAP_ANY_EDGE_GRAYMAP)
        type_requested |= 1;

    if (*type & FS_MAP_ANY_GRAYMAP)
        type_requested |= 2;

    if (*type & FS_MAP_BITMAP)
        type_requested |= 4;

    if (*type & FS_MAP_DISTANCEFIELD)
        type_requested = 8;

    senv = (SENV *)(STATE.cur_sfnt->senv);
    if (senv)
        rbv = senv->render_behavior;

    /* first resolt the rasterizer */
    switch (type_requested)
    {
    case 1: /* Edge only */
        *rasterizer = 4; /* Edge */
        break;

    case 2: /* Graymap only */
        *rasterizer = 2; /* graymap */
        break;

    case 3: /* Edge or Graymap */
    {
#ifdef FS_EDGE_RENDER
        if ( (rbv & DOEDGE) )
        {
            *rasterizer = 4; /* Edge */
            break;
        }
#endif
        *rasterizer = 2; /* graymap */
        break;
    }

    case 4: /* Bitmap */
        *rasterizer = 1; /* bitmap */
        break;

    case 5:    /* Edge or Bitmap */
    {
        if ( !(STATE.flags & FLAGS_HINTS_OFF))
        {
            if ( !(rbv & GRIDFIT) )
                FS_set_flags(_PS_ FLAGS_HINTS_OFF);
        }
#ifdef FS_EDGE_RENDER
        if ( (rbv & DOGRAY))
        {
            *rasterizer = 4; /* Edge */
            break;
        }
#endif
        *rasterizer = 1; /* bitmap */
        break;
    } /* end of case 5 */

    case 6: /* Graymap or Bitmap */
    {
        if ( !(STATE.flags & FLAGS_HINTS_OFF))
        {
            if ( !(rbv & GRIDFIT) )
                FS_set_flags(_PS_ FLAGS_HINTS_OFF);
        }
#ifdef FS_GRAYMAPS
        if ( (rbv & DOGRAY) )
        {
            *rasterizer = 2; /* graymap */
            break;
        }
#endif
        *rasterizer = 1; /* bitmap */
        break;
    }

    case 7: /* Edge or Gray or Bitmap */
    {
        if ( !(STATE.flags & FLAGS_HINTS_OFF))
        {
            if ( !(rbv & GRIDFIT) )
                FS_set_flags(_PS_ FLAGS_HINTS_OFF);
        }
#ifdef FS_EDGE_RENDER
        if ( ( ( rbv & DOGRAY ) && ( rbv & DOEDGE ) ) )
        {
            *rasterizer = 4; /* Edge */
            break;
        }
#endif
#ifdef FS_GRAYMAPS
        if ( rbv & DOGRAY )
        {
            *rasterizer = 2; /* graymap */
            break;
        }
#endif
        *rasterizer = 1; /* bitmap */
        break;
    }
    case 8: /* distance field only */
    {
        *rasterizer = 8; /* distance field */
        *type = FS_MAP_DISTANCEFIELD;
        return;
    }

    default:
        break;
    }    /* end of switch */

    /* now further resolve the type */
    if (*rasterizer == 1)
        *type = FS_MAP_BITMAP;
    else if (*rasterizer == 2)
    {
        if (*type & FS_MAP_GRAYMAP4)
            *type = FS_MAP_GRAYMAP4;
        else if (*type & FS_MAP_GRAYMAP8)
            *type = FS_MAP_GRAYMAP8;
        else
            *type = FS_MAP_GRAYMAP2;
    }
    else if (*rasterizer == 4)
    {
        if (*type & FS_MAP_EDGE_GRAYMAP4)
            *type = FS_MAP_EDGE_GRAYMAP4;
        else if (*type & FS_MAP_EDGE_GRAYMAP8)
            *type = FS_MAP_EDGE_GRAYMAP8;
        else if (*type & FS_MAP_EDGE_GRAYMAP2)
            *type = FS_MAP_EDGE_GRAYMAP2;
        else if (*type & FS_MAP_EDGE_RGBv8)
            *type = FS_MAP_EDGE_RGBv8;
        else if (*type & FS_MAP_EDGE_RGBv4)
            *type = FS_MAP_EDGE_RGBv4;
        else if (*type & FS_MAP_EDGE_RGBh8)
            *type = FS_MAP_EDGE_RGBh8;
        else if (*type & FS_MAP_EDGE_RGBh4)
            *type = FS_MAP_EDGE_RGBh4;
    }
}

/**
 * get a glyphmap which might be an icon or a bitmap or a graymap
 * depending on the type requested.
 */
FS_GLYPHMAP *FSS_get_glyphmap(_DS_ FS_ULONG id, FS_USHORT type)
{
    FS_GLYPHMAP *map = 0;
    FS_USHORT index; /* index ignored for user outline */
    FS_USHORT rasterizer = 0;

#ifdef FS_EXTERNAL_OUTLINE
    index = 0;
    if (!STATE.user_outline)
#endif
    {
        /* character id to index */
        index = map_char(_PS_ id, 0);
        if (STATE.error)
            return 0;

        if (check_sfnt(_PS0_))
            return 0;
#ifdef FS_HINTS
        if ((STATE.cur_lfnt->fontflags & FONTFLAG_STIK) != FONTFLAG_STIK &&
            (STATE.cur_lfnt->fnt_type != PFR_TYPE) &&
            (STATE.cur_sfnt->ref_lines[RTGA_BEEN_THERE] == 0))
            get_ref_lines(_PS0_); /* get RTGAH ref lines since not already done */
#endif
    }

#ifdef FS_ICONS
    {
        TTF *ttf = (TTF *)(STATE.cur_lfnt->fnt);

        /* if index is an icon index, look for icons */
        if (index >= ttf->maxp->numGlyphs)
        {
            if (type & FS_MAP_VECTOR_ICON)
            {
                map = get_icon(_PS_ index, FS_MAP_VECTOR_ICON);
                if (map)
                    return map;
            }
            if (type & FS_MAP_RASTER_ICON)
            {
                map = get_icon(_PS_ index, FS_MAP_RASTER_ICON);
                if (map)
                    return map;
            }
        }
    }
#endif /* FS_ICONS */

    resolve_glyphmap_type(_PS_ &type, &rasterizer);

    switch (rasterizer)
    {
    case 1: /* Bitmap */
#if defined(FS_BITMAPS) || defined(FS_EMBEDDED_BITMAP)
        map = (FS_GLYPHMAP *)get_bitmap(_PS_ id, index);
        return map;
#else
        break;
#endif

    case 2: /* Graymap only */
#if defined(FS_GRAYMAPS) || defined(FS_EMBEDDED_BITMAP)
        map = (FS_GLYPHMAP *)get_graymap(_PS_ id, index, (FS_USHORT)(type),0,0);
        return map;
#else
        break;
#endif

    case 4: /* Edge only */
#ifdef FS_EDGE_RENDER
        map = get_ADF_graymap(_PS_ id, index, (FS_USHORT)(type),0,0);
        return map;
#else
        break;
#endif


    case 8: /* distance field only */
    {
#ifdef FS_EDGE_RENDER
        map = (FS_GLYPHMAP *)get_distance_field(_PS_ id, index,0,0);
        return map;
#else
        STATE.error = ERR_FS_EDGE_RENDER_UNDEF;
        return 0;
#endif
    }
    default:
        break;
    }    /* end of switch */


    if (!STATE.error)
    {
        STATE.error = ERR_NOT_SUPPORTED;
        return map;
    }
    else
        return 0;
}

/*lint -e529  Warning -- Symbol 'rasterizer' not subsequently referenced */
/*lint -e438  Warning -- last value assigned to 'rasterizer' not used    */
/**
* get the glyphmap bounding box that corresponds to a particular id and type
* either retrieve it from cache or make it from an outline in the font.
*/
FS_BOUNDINGBOX *FSS_get_boundingbox(_DS_ FS_ULONG id, FS_USHORT type)
{
    FS_GLYPHMAP *gmap;
    FS_BOUNDINGBOX *bbox;
    FS_LONG size;

#ifdef FS_CACHE_BOUNDINGBOX
    FS_USHORT rasterizer=0;
    FS_USHORT index;

    /* character id to index */
    index = map_char(_PS_ id, 0);
    if (STATE.error)
        return NULL;

    if (check_sfnt(_PS0_))
        return NULL;

    resolve_glyphmap_type(_PS_ &type, &rasterizer);
    if (rasterizer > 8)
        STATE.error = ERR_NOT_SUPPORTED;

    /* first check the cache */
    bbox = (FS_BOUNDINGBOX *)find_bbox_in_cache(_PS_ index, type);
    if (bbox)
        return bbox;
#endif
    STATE.error = SUCCESS;

    /* otherwise make the bounding box */
    gmap = FSS_get_glyphmap(_PS_ id, type);
    if (gmap == NULL)
        return NULL;  /* with STATE.error set */

    bbox = FSS_malloc(_PS_ sizeof(FS_BOUNDINGBOX));
    if (bbox == NULL)
        return NULL;  /* with STATE.error set */

    size = offsetof(FS_GLYPHMAP, bits);
    SYS_MEMCPY(bbox, gmap, size);
    bbox->cache_ptr = NULL;
    bbox->bits[0] = 0;
    bbox->size = sizeof(FS_BOUNDINGBOX);

    FSS_free_char(_PS_ gmap);

#ifdef FS_CACHE_BOUNDINGBOX
    /* save bounding box to cache */
    save_bbox_to_cache(_PS_ index, type, bbox);
#endif

    return bbox;
}
/*lint +e529  Warning -- Symbol 'rasterizer' not subsequently referenced */
/*lint +e438  Warning -- last value assigned to 'rasterizer' not used    */

/**
* get a phased graymap by either retrieving it from
* cache or making it from an outline in the font.
*/
FS_GLYPHMAP *FSS_get_phased(_DS_ FS_ULONG id, FS_USHORT type, FS_FIXED posx, FS_FIXED posy)
{
#ifdef FS_PHASED
    FS_GLYPHMAP *map = 0;
    FS_SHORT cspx = (FS_SHORT)((posx & 0xFFFF) >> 14); /* current subpixel position index in x */
    FS_SHORT cspy = (FS_SHORT)((posy & 0xFFFF) >> 14); /* current subpixel position index in y */
    FS_USHORT index; /* index ignored for user outline */
    FS_USHORT rasterizer = 0;

#ifdef FS_EXTERNAL_OUTLINE
    index = 0;
    if (!STATE.user_outline)
#endif
    {
        /* character id to index */
        index = map_char(_PS_ id, 0);
        if (STATE.error)
            return 0;

        if (check_sfnt(_PS0_))
            return 0;
#ifdef FS_HINTS
        if ((STATE.cur_lfnt->fontflags & FONTFLAG_STIK) != FONTFLAG_STIK &&
            (STATE.cur_lfnt->fnt_type != PFR_TYPE) &&
            (STATE.cur_sfnt->ref_lines[RTGA_BEEN_THERE] == 0))
            get_ref_lines(_PS0_); /* get RTGAH ref lines since not already done */
#endif
    }

#ifdef FS_ICONS
    {
        TTF *ttf = (TTF *)(STATE.cur_lfnt->fnt);

        /* if index is an icon index, look for icons */
        if (index >= ttf->maxp->numGlyphs)
        {
            if (type & FS_MAP_VECTOR_ICON)
            {
                map = get_icon(_PS_ index, FS_MAP_VECTOR_ICON);
                if (map)
                    return map;
            }
            if (type & FS_MAP_RASTER_ICON)
            {
                map = get_icon(_PS_ index, FS_MAP_RASTER_ICON);
                if (map)
                    return map;
            }
        }
    }
#endif /* FS_ICONS */

    resolve_glyphmap_type(_PS_ &type, &rasterizer);

    switch (rasterizer)
    {
    case 1: /* Bitmap */
#if defined(FS_BITMAPS) || defined(FS_EMBEDDED_BITMAP)
        map = (FS_GLYPHMAP *)get_bitmap(_PS_ id, index);
        return map;
#else
        break;
#endif

    case 2: /* Graymap only */
#if defined(FS_GRAYMAPS) || defined(FS_EMBEDDED_BITMAP)
        map = (FS_GLYPHMAP *)get_graymap(_PS_ id, index, (FS_USHORT)(type),cspx,cspy);
        return map;
#else
        break;
#endif

    case 4: /* Edge only */
#ifdef FS_EDGE_RENDER
        map = get_ADF_graymap(_PS_ id, index, (FS_USHORT)(type),cspx,cspy);
        return map;
#else
        break;
#endif


    case 8: /* distance field only */
    {
#ifdef FS_EDGE_RENDER
        map = (FS_GLYPHMAP *)get_distance_field(_PS_ id, index,cspx,cspy);
        return map;
#else
        STATE.error = ERR_FS_EDGE_RENDER_UNDEF;
        return 0;
#endif
    }
    default:
        break;
    }    /* end of switch */


    if (!STATE.error)
    {
        STATE.error = ERR_NOT_SUPPORTED;
        return map;
    }
    else
        return 0;

#else   /* FS_PHASED not defined */
    posx = posx;
    posy = posy;
    type = type;
    id = id;
    STATE.error = ERR_PHASED_UNDEF;
    return 0;
#endif  /* FS_PHASED */
}

/**
 * if the character was cached, decrement the characters cache reference counter
 * and possibly the scaled font (sfnt) active count, then free the character data.
 */
FS_LONG FSS_free_char(_DS_ FS_VOID *p)
{

#ifdef FS_CACHE
    if (p)
    {
        /* we do assume all char objects have the cache_ptr in the same position */
        CACHE_ENTRY *cp = (CACHE_ENTRY *)(((FS_BITMAP *)p)->cache_ptr);
        if (cp)
        {
            SFNT *sfnt = (SFNT *)(cp->sfnt);
            /* it's been cached ... decrement characters usage count */
            if (cp->ref_counter > 0)
                cp->ref_counter--;

            /* if the character has no references, the sfnt has one less active character */
            /* but the font's cached_count is unchanged -- the character is still cached */
            if (sfnt && cp->ref_counter == 0 && sfnt->active_count > 0)
                sfnt->active_count--;

            return SUCCESS;
        }
        else
        {
            /* it's never been cached ... just free it */
            return FSS_free(_PS_ p);
        }
    }
    else
#endif
        return FSS_free(_PS_ p);
}

static FS_LONG FS_STATE_done(FS_STATE *sp)
{
    ADDEDFNT *fnt, *fnt_next;

#ifdef FS_EDGE_TECH
    stopADF(sp);
#endif

    for (fnt = sp->name_map; fnt; fnt = fnt_next)
    {
        fnt_next = fnt->next;

        if (fnt->add_count > 1)
        {
            /* ideally, all fonts would be deleted by now */
            /* the fact that we are in this loop means a leak of sorts */
            fnt->add_count = 1;
        }
        FSS_delete_font(sp, fnt->name);
    }

    sp->cur_lfnt = NULL;
    sp->cur_sfnt = NULL;

    /* clear the current typeset */
    TYPESET_destroy(sp, &sp->cur_typeset);

#ifdef FS_MAPPED_FONTS
    /* kill the mapped fonts */
    while (sp->mappedfnts)
    {
        MF_delete_mapped_font(sp, sp->mappedfnts->path);
    }
#endif
#ifdef FS_EXTERNAL_OUTLINE
    if (sp->user_outline)
    {
        FSS_free_char(sp, sp->user_outline);
        sp->user_outline = NULL;
    }
#endif
    return sp->error = SUCCESS;
}

/****************************************************************/
FS_LONG FSS_exit(_DS0_)
{
#ifdef FS_STATS
#ifdef FS_CACHE_ADVANCE
    FS_PRINTF(("%% cached_advance   =%9lu\n", (long)cached_advance));
#endif
#ifdef FS_CACHE_BOUNDINGBOX
    FS_PRINTF(("%% cached_bbox      =%9lu\n", (long)cached_bbox));
#endif
    FS_PRINTF(("%% made_outl        =%9lu\n", (long)made_outl));
    FS_PRINTF(("%% cached_outl      =%9lu\n", (long)cached_outl));
    FS_PRINTF(("%% made_bmap        =%9lu\n", (long)made_bmap));
    FS_PRINTF(("%% cached_bmap      =%9lu\n", (long)cached_bmap));
    FS_PRINTF(("%% embedded_bmap    =%9lu\n", (long)embedded_bmap));
    FS_PRINTF(("%% made_gmap        =%9lu\n", (long)made_gmap));
    FS_PRINTF(("%% cached_gmap      =%9lu\n", (long)cached_gmap));
    FS_PRINTF(("%% uncached_too_big =%9lu\n", (long)uncached_too_big));
    FS_PRINTF(("%% collisions       =%9lu\n", (long)collisions));
    FS_PRINTF(("%% num_composite_chars = %9lu\n", (long)num_composite_chars));
    FS_PRINTF(("%% num_composite_hints = %9lu\n", (long)num_composite_hints));
    FS_PRINTF(("%% size_composite_char = %9lu\n", (long)size_composite_char));
    FS_PRINTF(("%% cache_calls      =%9lu\n", (long)cache_calls));
#ifdef FS_DEBUG
    FS_PRINTF(("%% cache_probes     =%9lu\n", (long)cache_probes));
#endif
#ifdef FS_OPENVG
    FS_PRINTF(("%% made_openvg      =%9lu\n", (long)made_openvg));
#ifdef FS_CACHE_OPENVG
    FS_PRINTF(("%% cached_openvg    =%9lu\n", (long)cached_openvg));
#endif
#endif
#endif

    FS_STATE_done(_PS0_);

#ifdef FS_CACHE
    {
        /* kill all cached objects - regardless of their ref_counter */

        int i;
        CACHE_ENTRY *p, *next;

        for (i = 0; i < CACHE_MOD; i++)
        {
            p = (CACHE_ENTRY *)(STATE.server->cache[i]);
            while (p)
            {
#ifdef FS_MEMORY_LEAK_CHECK
                if (p->ref_counter)
                {
                    /* this glyph has outstanding references */
                    FS_PRINTF(("on exit: cache_entry %p ref_counter=%ld\n", (FS_VOID *)p, p->ref_counter));
                    /* this should never happen */
                    /* save this code -- it allows Purify to find leak source
                    p = p->next;
                    continue;
                    save the above code for Purify help in finding leaks */
                }
#endif
                next = p->next;
                FSS_free(_PS_ (FS_BYTE *)(p->data));    /* first the data element */
#ifdef FS_NO_FS_LUMP
                FSS_free(_PS_ p);        /* now the cache entry itself */
#else
                CACHE_ENTRY_free(_PS_ p);
#endif
                p = (CACHE_ENTRY *)(next);
            }
            STATE.server->cache[i] = 0;        /* mark the cache row as empty */
        }
    }
#endif /* FS_CACHE */

    {
        /* kill the scaled fonts - regardless of ref_count or active_count */

        SFNT *sfnt, *sfnt_next;

        for (sfnt = (SFNT *)(STATE.server->scaled_fonts); sfnt; sfnt = (SFNT *)(sfnt_next))
        {
            sfnt_next = sfnt->next;
#ifdef FS_MEMORY_LEAK_CHECK
            if (sfnt->ref_count)
            {
                /* somebody has this as current scaled font */
                FS_PRINTF(("on exit: sfnt %p ref_count=%ld\n", (FS_VOID *)sfnt, sfnt->ref_count));
                /* this should never happen */
            }
#endif
#if defined(FS_DEBUG) || defined(FS_MEMORY_LEAK_CHECK)
            if (sfnt->active_count)
            {
                /* this scaled font has active characters */
                FS_PRINTF(("on exit: sfnt %p active_count=%lu\n", (FS_VOID *)sfnt,
                           (long)sfnt->active_count));
                /* this should never happen */
            }
#endif
#ifdef FS_MEMORY_LEAK_CHECK
            if (sfnt->cache_count)
            {
                /* this scaled font has cached characters */
                FS_PRINTF(("on exit: sfnt %p cache_count=%ld\n", (FS_VOID *)sfnt, sfnt->cache_count));
                /* this should never happen */
            }
#endif
            if ((LFNT *)(sfnt->lfnt))
            {
                SENV *senv;
                FS_BYTE fnt_type;
                senv = (SENV *)(sfnt->senv);
                if (senv)
                {
                    fnt_type = sfnt->lfnt->fnt_type;
                    if ((fnt_type == TTF_TYPE) || (fnt_type == TTC_TYPE) ||
                            (fnt_type == CFF_TYPE))
                        delete_key(_PS_ (FS_VOID *)(sfnt->senv->ttkey));
                    else
                        FSS_free(_PS_ (FS_VOID *)(sfnt->senv->ttkey));
                    FSS_free(_PS_ sfnt->senv);
                }
            }
            FSS_free(_PS_ sfnt);
        }
    }

    {
        /* kill the loaded_fonts */

        LFNT *lfnt, *lfnt_next;

#ifdef FS_MEMORY_LEAK_CHECK
        {
            TABLE_PTR_REC *rec;

            for (rec = (TABLE_PTR_REC *)(STATE.server->table_ptrs); rec; rec = (TABLE_PTR_REC *)(rec->next))
            {
                if (rec->ref_count)
                {
                    /* this table has outstanding references */
                    FS_PRINTF(("on exit: rec %p ref_count=%ld\n", (FS_VOID *)rec, rec->ref_count));
                    /* this should never happen */
                }
            }
        }
#endif
        kill_table_ptrs(_PS0_);

        for (lfnt = (LFNT *)(STATE.server->loaded_fonts); lfnt; lfnt = (LFNT *)(lfnt_next))
        {
#ifdef FS_MEMORY_LEAK_CHECK
            if (lfnt->fntset_refs)
            {
                /* this lfnt is still referenced by a fntset */
                FS_PRINTF(("on exit: lfnt %p fntset_refs=%ld\n", (FS_VOID *)lfnt, lfnt->fntset_refs));
                /* this should never happen */
            }
#endif
            lfnt_next = lfnt->next;
            delete_kern(_PS_ lfnt);
            unload_fnt(_PS_ lfnt);
#ifdef FS_CACHE_CMAP
            if (lfnt->cmap_cache) FSS_free(_PS_ (FS_CMAP_CACHE *)(lfnt->cmap_cache));
#endif
            FSS_free(_PS_ (FILECHAR *)(lfnt->name));
            FSS_free(_PS_ (FILECHAR *)(lfnt->path));
            FSS_free(_PS_ lfnt);
        }
    }

    {
        /* kill the font sets */

        FNTSET *fntset, *fntset_next;

        for (fntset = (FNTSET *)(STATE.server->font_sets); fntset; fntset = (FNTSET *)(fntset_next))
        {
            CFNT *cfnt;
            FS_USHORT nc;

#ifdef FS_MEMORY_LEAK_CHECK
            if (fntset->ref_count)
            {
                /* this fontset is a current fontset somewhere */
                FS_PRINTF(("on exit: fntset %p ref_count=%ld\n", (FS_VOID *)fntset, fntset->ref_count));
                /* this should never happen */
            }
            if (fntset->count_added)
            {
                /* this fontset has not been deleted as many times as added */
                FS_PRINTF(("on exit: fntset %p count_added=%ld\n", (FS_VOID *)fntset, fntset->count_added));
                /* this should never happen */
            }
#endif

            fntset_next = fntset->next;

            FSS_free(_PS_ (FILECHAR *)(fntset->name));
            FSS_free(_PS_ (FILECHAR *)(fntset->path));

            for (cfnt = fntset->cfnt, nc = 0; nc < fntset->num_fonts; cfnt++, nc++)
            {
                cfnt->name = 0;
            }
            FSS_free(_PS_ (CFNT *)(fntset->cfnt));
            if ((FNTSET_OT_TABLE *)(fntset->cmap))
                FSS_free(_PS_ (FNTSET_OT_TABLE *)(fntset->cmap));
            if ((FNTSET_OT_TABLE *)(fntset->gdef))
                FSS_free(_PS_ (FNTSET_OT_TABLE *)(fntset->gdef));
            if ((FNTSET_OT_TABLE *)(fntset->gsub))
                FSS_free(_PS_ (FNTSET_OT_TABLE *)(fntset->gsub));
            if ((FNTSET_OT_TABLE *)(fntset->gpos))
                FSS_free(_PS_ (FNTSET_OT_TABLE *)(fntset->gpos));
            FSS_free(_PS_ fntset);
        }
    }
#ifdef FS_CONTOUR_WINDING_DETECTION
    if ((FS_LONG *)(STATE.server->nextLoopTX))
    {
        FSS_free(_PS_ (FS_LONG *)(STATE.server->nextLoopTX));
        STATE.server->nextLoopTX = 0;
    }
#endif
    {
        /*  free the workspace */
        if (STATE.server->workspaceSize > 0)
        {
            FSS_free(_PS_ (FS_BYTE *)(STATE.server->workspace));
            STATE.server->workspace = 0;
        }

#if defined(FS_BITMAPS)
        delete_tlist(_PS_ (TLIST *)(STATE.server->tlist));
        delete_tlist(_PS_ (TLIST *)(STATE.server->drop));
#endif
    }

#ifndef FS_NO_FS_LUMP
    {
        while (STATE.server->list_of_lumps)
        {
            FS_MASTER_LUMP *curr = (FS_MASTER_LUMP *)(STATE.server->list_of_lumps);
            STATE.server->list_of_lumps = curr->next;
            FSS_free(_PS_ curr);
        }
        STATE.server->CACHE_ENTRY_chain = NULL;
    }
#endif

#ifdef FS_GRAYMAPS
    {
        if (STATE.server->raster)
        {
            RASTER *r = (RASTER *)(STATE.server->raster);
            if (r->areasize)
                FSS_free(_PS_ r->areas);
            FSS_free(_PS_ r);
        }
    }
#endif

    {
        /* free the FsScratchSpace */
        FsScratchSpace_done(&STATE.server->scratch, FS_state_ptr);
    }

#ifdef FS_INT_MEM
    __term_heap(_PS0_);
#endif

#ifdef FS_DEBUG
    FS_PRINTF(("%% exit allocated   =%9lu\n", (long)STATE.server->allocated));
#endif

    /* free up iType's internal server area */
#ifndef FS_INT_MEM
    SYS_FREE(STATE.server);
#else
    if (STATE.server->iTypeOwnsHeap == true)
        SYS_FREE(STATE.server);
    else
        SYS_MEMSET(STATE.server, 0, sizeof(FS_SERVER));
#endif
    STATE.server = NULL;

    return SUCCESS;
}
/****************************************************************/
#ifdef FS_DUMP
FS_VOID dump_scaled_fonts(_DS_ FILECHAR *str)
{
    SFNT *p;

    FS_PRINTF(("%s\nscaled_fonts\n", str));
    for (p = (SFNT *)(STATE.server->scaled_fonts); p; p = (SFNT *)(p->next))
    {
        FS_FIXED *s = &p->user_scale[0];

        FS_PRINTF(("\t%p lfnt=%p '%s' scale=%ld %ld %ld %ld\n",
                   (FS_VOID *)p, (FS_VOID *)p->lfnt,
                   p->lfnt->name,
                   (long)s[0], (long)s[1], (long)s[2], (long)s[3]));
    }
    FS_PRINTF(("cur_sfnt = %p\n\n", (FS_VOID *)(STATE.cur_sfnt)));
}
#endif
/****************************************************************/
#ifdef FS_DUMP
FS_VOID dump_loaded_fonts(_DS_ FILECHAR *s)
{
    LFNT *p;

    FS_PRINTF(("%s\nloaded fonts\n", s));
    for (p = (LFNT *)(STATE.server->loaded_fonts); p; p = (LFNT *)(p->next))
    {
        FILECHAR *name = (FILECHAR *)(p->name);
        FS_PRINTF(("\t%p '%s' ttf=%p\n", (FS_VOID *)p, name, (FS_VOID *)(p->fnt)));
    }
    FS_PRINTF(("cur_lfnt = %p\n\n", (FS_VOID *)(STATE.cur_lfnt)));
}
#endif

#ifdef FS_LINKED_FONTS
static FNTSET_OT_TABLE *
FsLtt_set_FNTSET_OT_TABLE(_DS_ FsLttOtTable *t)
{
    STATE.error = SUCCESS;

    if (t->offset == 0)
    {
        return NULL;    /* get from component font 0 */
    }
    else
    {
        FNTSET_OT_TABLE *ot;

#ifdef FS_MEM_DBG
        STATE.memdbgid = "FNTSET_OT_TABLE";
#endif
        ot = (FNTSET_OT_TABLE *)FSS_calloc(_PS_ sizeof(FNTSET_OT_TABLE));
        if (ot)
        {
            ot->offset = t->offset;
            ot->size = t->size;
        }
        return ot;
    }
}
#endif /* FS_LINKED_FONTS */

static FS_LONG
FS_STATE_add_named_FNTSET(_DS_ FILECHAR *name, FNTSET *fntset)
{
    ADDEDFNT *fnt;
    FS_ULONG len = SYS_STRLEN(name);

#ifdef FS_MEM_DBG
    STATE.memdbgid = "ADDEDFNT";
#endif
    fnt = FSS_malloc(_PS_ sizeof(ADDEDFNT));
    if (!fnt)
    {
        return STATE.error = ERR_MALLOC_FAIL;
    }
#ifdef FS_MEM_DBG
    STATE.memdbgid = "fnt->name";
#endif
    fnt->name = FSS_malloc(_PS_ len + 1);
    if (!fnt->name)
    {
        FSS_free(_PS_ fnt);
        return STATE.error = ERR_MALLOC_FAIL;
    }
    SYS_STRNCPY(fnt->name, name, len);
    fnt->name[len] = '\0';

    fnt->set = fntset;

    fnt->add_count = 1;

    fnt->next = STATE.name_map;

    STATE.name_map = fnt;

    fntset->count_added++;

    return SUCCESS;
}

static FS_ULONG
FNTSET_checksum(FS_BYTE *memptr)
{
    FS_ULONG checksum = 0;
    FS_LONG i = FNTSET_CHECKSUM_LENGTH;
    FS_USHORT *pmemptr = (FS_USHORT *)memptr;
    while (i-- > 0)
    {
        checksum += (FS_ULONG)(*pmemptr++);
    }
    return checksum;
}

static FS_BOOLEAN
FNTSET_same_file(FNTSET *fntset,
                 FILECHAR *path,
                 FS_BYTE *memptr,
                 FS_ULONG data_offset,
                 FS_ULONG data_length)
{
    if (fntset->path && path)
    {
        FILECHAR *s;
        s = (FILECHAR *)(fntset->path);
        /* these shenanigans remove Linux "possible NULL" warning */
        if (!s) /* this'll never happen */
            return 0;

        return (!SYS_STRCMP(s, path) &&
                (fntset->data_offset == data_offset) &&
                (fntset->data_length == data_length));
    }
    else if (fntset->path || path)
    {
        return 0;
    }
    else if (!fntset->memptr || !memptr)
    {
        return 0;
    }
    else if ((fntset->memptr == memptr) &&
             (fntset->data_offset == data_offset) &&
             (fntset->data_length == data_length))
    {
        FS_ULONG checksum = FNTSET_checksum(memptr);
        return (checksum == fntset->checksum);
    }
    else
        return 0;
}

static FS_BOOLEAN
FNTSET_same_font(FNTSET *fntset,
                 FILECHAR *path,
                 FS_BYTE *memptr,
                 FS_ULONG index,
                 FS_ULONG data_offset,
                 FS_ULONG data_length)
{
    if (FNTSET_same_file(fntset, path, memptr, data_offset, data_length))
    {
        if (fntset->num_fonts == 1)
        {
            CFNT *cfnt = (CFNT *)(fntset->cfnt);
            LFNT *lfnt = (LFNT *)(cfnt->lfnt);

            if (lfnt->fnt_type == TTC_TYPE)
            {
                if (lfnt->index == index)
                {
                    return 1;
                }
                else return 0;
            }
            else return 1;  /* "index" is irrelevant */
        }
        else return 1;  /* "index" is irrelevant */
    }
    else return 0;
}

static FNTSET *
FS_SERVER_find_FNTSET(_DS_
                      FILECHAR *path,
                      FS_BYTE *memptr,
                      FS_ULONG index,
                      FS_ULONG data_offset,
                      FS_ULONG data_length)
{
    FNTSET *fntset;

    for (fntset = (FNTSET *)(STATE.server->font_sets);
            fntset;
            fntset = (FNTSET *)(fntset->next))
    {
        if (FNTSET_same_font(fntset, path, memptr,
                             index, data_offset, data_length))
        {
            return fntset;
        }
    }
    return NULL;
}

static FS_VOID load_cmap_cache(_DS_ LFNT *lfnt)
{
#ifdef FS_CACHE_CMAP
    TTF *ttf = (TTF *)(lfnt->fnt);

    if (lfnt->cmap_cache) return; /* already loaded */

    if (ttf && ttf->maxp)
    {
        FS_USHORT n;
        FS_USHORT m, last_m;
        FS_VOID *p;

        n = ttf->maxp->numGlyphs;
#ifdef FS_ICONS
        n += (FS_USHORT)ttf->num_icons;
#endif
        n /= 10;

        if (n < 256) n = 256;

        last_m = 1;
        for (m = 1; m < n; m += m)
        {
            last_m = m;
        }
        if ((m != n) && ((n - last_m) < (m - n)))
        {
            m = last_m;
        }
#ifdef FS_MEM_DBG
        STATE.memdbgid = "CMAP_CACHE";
#endif
        p = FSS_calloc(_PS_ m * sizeof(FS_CMAP_CACHE));
        if (p)
        {
            lfnt->cmap_cache = p;
            lfnt->cmap_cache_mod = m - 1;
        }
    }
#else
    FS_state_ptr = FS_state_ptr;
    lfnt = lfnt;
#endif /* FS_CACHE_CMAP */
}

static FS_BOOLEAN
LFNT_same_file(LFNT *lfnt,
               FILECHAR *path,
               FS_BYTE *memptr,
               FS_ULONG data_offset,
               FS_ULONG data_length)
{
    if (lfnt->path && path)
    {
        FILECHAR *s;
        s = (FILECHAR *)(lfnt->path);
        /* these shenanigans remove Linux "possible NULL" warning */
        if (!s) /* this'll never happen */
            return 0;

        return (!SYS_STRCMP(s, path) &&
                (lfnt->data_offset == data_offset) &&
                (lfnt->data_length == data_length));
    }
    else if (lfnt->path || path)
    {
        return 0;
    }
    else if (!lfnt->memptr || !memptr)
    {
        return 0;
    }
    else if ((lfnt->memptr == memptr) &&
             (lfnt->data_offset == data_offset) &&
             (lfnt->data_length == data_length))
    {
        FS_ULONG checksum = FNTSET_checksum(memptr);
        return (checksum == lfnt->checksum);
    }
    else
        return 0;
}

static FS_BOOLEAN
LFNT_same_font(LFNT *lfnt,
               FILECHAR *path,
               FS_BYTE *memptr,
               FS_ULONG index,
               FS_ULONG data_offset,
               FS_ULONG data_length)
{
    if (LFNT_same_file(lfnt, path, memptr, data_offset, data_length))
    {
        if (lfnt->fnt_type == TTC_TYPE)
        {
            if (lfnt->index == index)
            {
                return 1;
            }
            else return 0;
        }
        else return 1;  /* "index" is irrelevant */
    }
    else return 0;
}

static LFNT *
FS_SERVER_find_LFNT(_DS_
                    FILECHAR *path,
                    FS_BYTE *memptr,
                    FS_ULONG index,
                    FS_ULONG data_offset,
                    FS_ULONG data_length)
{
    LFNT *lfnt;

    for (lfnt = (LFNT *)(STATE.server->loaded_fonts);
         lfnt;
         lfnt = (LFNT *)(lfnt->next))
    {
        if (LFNT_same_font(lfnt, path, memptr,
                           index, data_offset, data_length))
        {
            return lfnt;
        }
    }
    return NULL;
}

static FS_LONG
FS_SERVER_load_LFNT(_DS_ LFNT * lfnt)
{
    if (load_fnt(_PS_ lfnt) != SUCCESS)
    {
        delete_lfnt(_PS_ lfnt);
        return STATE.error;
    }
    if ((lfnt->fnt_type == TTF_TYPE) && 
        (((TTF *)lfnt->fnt)->ttc_header))
    {
        lfnt->fnt_type = TTC_TYPE;
    }
    {
        FILECHAR *name;

        if ((lfnt->fnt_type == TTF_TYPE) || 
            (lfnt->fnt_type == TTC_TYPE) || 
            (lfnt->fnt_type == CFF_TYPE))
        {
            TTF *ttf = (TTF *)lfnt->fnt;

            name = ttf->name->font_name;
        }

#ifdef FS_PFRR
        else if (lfnt->fnt_type == PFR_TYPE)
        {
            PFR *pfr = (PFR *)lfnt->fnt;

            name = pfr->name;
        }
#endif /* FS_PFRR */

        else
        {
            name = NULL;
        }
        if(lfnt->name == NULL)
        {
            FILECHAR *dup;

#ifdef FS_MEM_DBG
            STATE.memdbgid = "lfnt->name";
#endif
            dup = FS_strdup(_PS_ name);
            if (!dup)
            {
                delete_lfnt(_PS_ lfnt);
                return STATE.error;
            }
            lfnt->name = dup;
        }
    }
    return STATE.error;
}

static LFNT *
FS_SERVER_create_LFNT(_DS_
                      FS_BYTE file_type,
                      FILECHAR *path,
                      FS_BYTE *memptr,
                      FS_ULONG index,
                      FS_ULONG data_offset,
                      FS_ULONG data_length)
{
    LFNT *lfnt;

#ifdef FS_MEM_DBG
    STATE.memdbgid = "LFNT";
#endif
    lfnt = FSS_calloc(_PS_ sizeof(LFNT));
    if (!lfnt)
    {
        return NULL;
    }
    if (path)
    {
        FILECHAR *dup;

#ifdef FS_MEM_DBG
        STATE.memdbgid = "lfnt->path";
#endif
        dup = FS_strdup(_PS_ path);
        if (!dup)
        {
            FSS_free(_PS_ lfnt);
            return NULL;
        }
        lfnt->path = dup;
    }
    else lfnt->path = NULL;

    lfnt->memptr = memptr;
    lfnt->data_offset = data_offset;
    lfnt->data_length = data_length;
    lfnt->checksum = 0;
    if (memptr)
    {
        lfnt->checksum = FNTSET_checksum(memptr); /* works for LFNT too */
    }

    lfnt->index = index;

    lfnt->fnt_type = file_type;

#ifdef FS_LOAD_UPFRONT
    FS_SERVER_load_LFNT(_PS_ lfnt);
#endif  /* FS_LOAD_UPFRONT */

    lfnt->fntset_refs = 0;

    lfnt->loading = 0;

    /* prepend to available fonts list */
    lfnt->next = STATE.server->loaded_fonts;
    STATE.server->loaded_fonts = lfnt;

#ifdef FS_CACHE_CMAP
    if ((lfnt->fnt_type == TTF_TYPE) || (lfnt->fnt_type == TTC_TYPE) ||
            (lfnt->fnt_type == CFF_TYPE))
    {
        load_cmap_cache(_PS_ lfnt);
    }
#endif

    return lfnt;
}

static LFNT *
FS_SERVER_get_LFNT(_DS_
                   FS_BYTE file_type,
                   FILECHAR *path,
                   FS_BYTE *memptr,
                   FS_ULONG index,
                   FS_ULONG data_offset,
                   FS_ULONG data_length)
{
    LFNT *lfnt;

    lfnt = FS_SERVER_find_LFNT(_PS_ path, memptr,
                               index, data_offset, data_length);

    if (lfnt)
    {
        return lfnt;
    }
    else
    {
        lfnt = FS_SERVER_create_LFNT(_PS_ file_type, path, memptr,
                                     index, data_offset, data_length);
        if (lfnt)
        {
            return lfnt;
        }
        else
        {
            return NULL;
        }
    }
}

static CFNT *
FS_SERVER_create_CFNT(_DS_
                      FS_BYTE file_type,
                      FILECHAR *path,
                      FS_BYTE *memptr,
                      FS_ULONG index,
                      FS_ULONG data_offset,
                      FS_ULONG data_length)
{
    LFNT *lfnt;

    lfnt = FS_SERVER_get_LFNT(_PS_ file_type, path, memptr,
                              index, data_offset, data_length);
    if (!lfnt)
    {
        return NULL;
    }
    else
    {
        CFNT *cfnt;

#ifdef FS_MEM_DBG
        STATE.memdbgid = "CFNT";
#endif
        cfnt = FSS_calloc(_PS_ sizeof(CFNT));
        if (!cfnt)
        {
            return NULL;
        }
        cfnt->name = lfnt->name;
        cfnt->lfnt = lfnt;
        lfnt->fntset_refs++;

        return cfnt;
    }
}

#ifdef FS_LINKED_FONTS

static FS_ULONG
FS_SERVER_init_CFNT(_DS_
                    CFNT *cfnt,
                    FS_BYTE file_type,
                    FILECHAR *path,
                    FS_BYTE *memptr,
                    FS_ULONG index,
                    FS_ULONG data_offset,
                    FS_ULONG data_length)
{
    LFNT *lfnt;

    lfnt = FS_SERVER_get_LFNT(_PS_ file_type, path, memptr,
                              index, data_offset, data_length);
    if (!lfnt)
    {
        return STATE.error;
    }
    else
    {
        cfnt->name = lfnt->name;
        cfnt->lfnt = lfnt;
        lfnt->fntset_refs++;

        return SUCCESS;
    }
}
#endif /* FS_LINKED_FONTS */

static FS_VOID
CFNT_delete(CFNT *cfnt)
{
    LFNT *lfnt;

    lfnt = (LFNT *)(cfnt->lfnt);

    lfnt->fntset_refs--;

    cfnt->name = 0;
}
#define DEFAULT_CFNT_RANGE_VALUE 0xFFFFFFFF;
static FNTSET *
FS_SERVER_create_FNTSET(_DS_
                        FILECHAR *path,
                        FS_BYTE *memptr,
                        FS_ULONG index,
                        FS_ULONG data_offset,
                        FS_ULONG data_length,
                        FILECHAR *lttPath)
{
    FNTSET *fntset;

    if (!path && !memptr)
    {
        STATE.error = ERR_FONT_NOT_FOUND;
        return NULL;
    }

#ifdef FS_MEM_DBG
    STATE.memdbgid = "FNTSET";
#endif

    fntset = FSS_calloc(_PS_ sizeof(FNTSET));
    if (!fntset)
    {
        return NULL;
    }
    if (path)
    {
        FILECHAR *dup;

#ifdef FS_MEM_DBG
        STATE.memdbgid = "fntset->path";
#endif

        dup = FS_strdup(_PS_ path);
        if (!dup)
        {
            FSS_free(_PS_ fntset);
            STATE.error = ERR_FONT_NOT_FOUND;
            return NULL;
        }
        fntset->path = dup;
    }
    else fntset->path = NULL;

    fntset->memptr = memptr;
    fntset->data_offset = data_offset;
    fntset->data_length = data_length;
    fntset->checksum = 0;
    if (memptr)
    {
        fntset->checksum = FNTSET_checksum(memptr);
    }

    {
        FS_BYTE file_type;

        file_type = fontfile_type(_PS_ path, memptr, data_offset);

        if (file_type == UNK_TYPE)
        {
            if (fntset->path) FSS_free(_PS_ (FILECHAR *)(fntset->path));
            FSS_free(_PS_ fntset);
            return NULL;
        }
        else if (file_type != LTT_TYPE)
        {
            /* single font set */

            CFNT *cfnt;

            cfnt = FS_SERVER_create_CFNT(_PS_ file_type,
                                         path, memptr, index,
                                         data_offset, data_length);
            if (!cfnt)
            {
                if (fntset->path) FSS_free(_PS_ (FILECHAR *)(fntset->path));
                FSS_free(_PS_ fntset);
                return NULL;
            }
            fntset->cfnt = cfnt;

            cfnt->bmp.range1 = DEFAULT_CFNT_RANGE_VALUE;
            cfnt->bmp.range2 = DEFAULT_CFNT_RANGE_VALUE;
            cfnt->smp.range1 = DEFAULT_CFNT_RANGE_VALUE;
            cfnt->smp.range2 = DEFAULT_CFNT_RANGE_VALUE;
            cfnt->sip.range1 = DEFAULT_CFNT_RANGE_VALUE;
            cfnt->sip.range2 = DEFAULT_CFNT_RANGE_VALUE;
            cfnt->ssp.range1 = DEFAULT_CFNT_RANGE_VALUE;
            cfnt->ssp.range2 = DEFAULT_CFNT_RANGE_VALUE;

            if(cfnt->name == NULL)
            {
                FS_SERVER_load_LFNT(_PS_ cfnt->lfnt);
                if (STATE.error)
                {
                    if (fntset->path) FSS_free(_PS_ (FILECHAR *)(fntset->path));
                    FSS_free(_PS_ fntset);
                    return NULL;
                }
                cfnt->name = cfnt->lfnt->name;
            }

            {
                FILECHAR *dup;

#ifdef FS_MEM_DBG
                STATE.memdbgid = "fntset->name";
#endif
                dup = FS_strdup(_PS_ (FILECHAR *)(cfnt->name));
                if (!dup)
                {
                    CFNT_delete(cfnt);
                    if (fntset->path)
                        FSS_free(_PS_ (FILECHAR *)(fntset->path));
                    FSS_free(_PS_ fntset);
                    return NULL;
                }
                fntset->name = dup;
            }

            fntset->num_fonts = 1;
            fntset->metric_font = cfnt->lfnt;

            fntset->name_offset = 0;
            fntset->name_size   = 0;
            fntset->maxp_offset = 0;
            fntset->os_2_offset = 0;
            fntset->head_offset = 0;
            fntset->hhea_offset = 0;
            fntset->vhea_offset = 0;
            fntset->post_offset = 0;

            fntset->cmap = NULL;

            fntset->gdef = NULL;
            fntset->gsub = NULL;
            fntset->gpos = NULL;

            fntset->ref_count = 0;
            fntset->count_added = 0;

            /* prepend to list of available font sets */
            fntset->next = STATE.server->font_sets;
            STATE.server->font_sets = fntset;

            return fntset;
        }
        else /* LTT_TYPE */
#ifdef FS_LINKED_FONTS
        {
            FsLtt *ltt;
            FS_USHORT cnt;
            FILECHAR *compPath;
            FILECHAR *compFile;
            FS_USHORT prev_start_index;
            CFNT *cfnt;
            FILECHAR *buffer;

#ifdef FS_MEM_DBG
            STATE.memdbgid = "compPath";
#endif
            compPath = (FILECHAR *)FsScratchSpace_reserve(&STATE.server->scratch, FS_state_ptr,
                       256 * sizeof(FILECHAR));
            if (compPath == 0 || STATE.error)
            {
                if (fntset->path) FSS_free(_PS_ (FILECHAR *)(fntset->path));
                FSS_free(_PS_ fntset);
                return NULL;
            }

            ltt = (FsLtt *)FsScratchSpace_reserve(&STATE.server->scratch, FS_state_ptr, sizeof(FsLtt));
            if (!ltt)
            {
                if (fntset->path) FSS_free(_PS_ (FILECHAR *)(fntset->path));
                FSS_free(_PS_ fntset);
                FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, compPath);
                return NULL;
            }

            if (FsLtt_init(_PS_ ltt))
            {
                if (fntset->path) FSS_free(_PS_ (FILECHAR *)(fntset->path));
                FSS_free(_PS_ fntset);
                FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, compPath);
                FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, ltt);
                return NULL;
            }

            if (memptr)
            {
                FS_LONG error;

                error = FsLtt_load(ltt, memptr, 0);
                if (error)
                {
                    FsLtt_done(ltt);
                    FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, ltt);
                    if (fntset->path) FSS_free(_PS_ (FILECHAR *)(fntset->path));
                    FSS_free(_PS_ fntset);
                    STATE.error = error;
                    FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, compPath);
                    return NULL;
                }
            }
            else if (path)
            {
                FS_FILE *fp;
                FS_LONG error;

                fp = FS_open(_PS_ path);
                if (!fp)
                {
                    FsLtt_done(ltt);
                    FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, ltt);
                    if (fntset->path) FSS_free(_PS_ (FILECHAR *)(fntset->path));
                    FSS_free(_PS_ fntset);
                    FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, compPath);
                    return NULL;
                }
                error = FsLtt_read(ltt, fp, 0);
                if (error)
                {
                    FS_close(_PS_ fp);
                    FsLtt_done(ltt);
                    FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, ltt);
                    if (fntset->path) FSS_free(_PS_ (FILECHAR *)(fntset->path));
                    FSS_free(_PS_ fntset);
                    FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, compPath);
                    STATE.error = error;
                    return NULL;
                }
                FS_close(_PS_ fp);
            }

            {
                FILECHAR *dup;

#ifdef FS_MEM_DBG
                STATE.memdbgid = "fntset->name";
#endif
                dup = FS_strdup(_PS_ ltt->name);
                if (!dup)
                {
                    FsLtt_done(ltt);
                    FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, ltt);
                    if (fntset->path) FSS_free(_PS_ (FILECHAR *)(fntset->path));
                    FSS_free(_PS_ fntset);
                    FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, compPath);
                    return NULL;
                }
                fntset->name = dup;
            }

            fntset->num_fonts = ltt->numComponents;

            fntset->name_offset = ltt->ttf_name.offset;
            fntset->name_size   = ltt->ttf_name.size;
            fntset->maxp_offset = ltt->maxp_off;
            fntset->os_2_offset = ltt->os_2_off;
            fntset->head_offset = ltt->head_off;
            fntset->hhea_offset = ltt->hhea_off;
            fntset->vhea_offset = ltt->vhea_off;
            fntset->post_offset = ltt->post_off;

            fntset->cmap = FsLtt_set_FNTSET_OT_TABLE(_PS_ & ltt->cmap);
            if (STATE.error != SUCCESS)
            {
                FSS_free(_PS_ (FILECHAR *)(fntset->name));
                FsLtt_done(ltt);
                FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, ltt);
                if (fntset->path) FSS_free(_PS_ (FILECHAR *)(fntset->path));
                FSS_free(_PS_ fntset);
                FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, compPath);
                return NULL;
            }

            fntset->gdef = FsLtt_set_FNTSET_OT_TABLE(_PS_ & ltt->gdef);
            if (STATE.error != SUCCESS)
            {
                if ((FNTSET_OT_TABLE *)(fntset->cmap))
                    FSS_free(_PS_ (FNTSET_OT_TABLE *)(fntset->cmap));
                FSS_free(_PS_ (FILECHAR *)(fntset->name));
                FsLtt_done(ltt);
                FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, ltt);
                if (fntset->path) FSS_free(_PS_ (FILECHAR *)(fntset->path));
                FSS_free(_PS_ fntset);
                FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, compPath);
                return NULL;
            }

            fntset->gsub = FsLtt_set_FNTSET_OT_TABLE(_PS_ & ltt->gsub);
            if (STATE.error != SUCCESS)
            {
                if ((FNTSET_OT_TABLE *)(fntset->gdef))
                    FSS_free(_PS_ (FNTSET_OT_TABLE *)(fntset->gdef));
                if ((FNTSET_OT_TABLE *)(fntset->cmap))
                    FSS_free(_PS_ (FNTSET_OT_TABLE *)(fntset->cmap));
                FSS_free(_PS_ (FILECHAR *)(fntset->name));
                FsLtt_done(ltt);
                FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, ltt);
                if (fntset->path) FSS_free(_PS_ (FILECHAR *)(fntset->path));
                FSS_free(_PS_ fntset);
                FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, compPath);
                return NULL;
            }

            fntset->gpos = FsLtt_set_FNTSET_OT_TABLE(_PS_ & ltt->gpos);
            if (STATE.error != SUCCESS)
            {
                if ((FNTSET_OT_TABLE *)(fntset->gsub))
                    FSS_free(_PS_ (FNTSET_OT_TABLE *)(fntset->gsub));
                if ((FNTSET_OT_TABLE *)(fntset->gdef))
                    FSS_free(_PS_ (FNTSET_OT_TABLE *)(fntset->gdef));
                if ((FNTSET_OT_TABLE *)(fntset->cmap))
                    FSS_free(_PS_ (FNTSET_OT_TABLE *)(fntset->cmap));
                FSS_free(_PS_ (FILECHAR *)(fntset->name));
                FsLtt_done(ltt);
                FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, ltt);
                if (fntset->path) FSS_free(_PS_ (FILECHAR *)(fntset->path));
                FSS_free(_PS_ fntset);
                FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, compPath);
                return NULL;
            }
            {
                FILECHAR *sf, *sb;
                SYS_STRNCPY(compPath, lttPath ? lttPath : "", 255);
                compPath[255] = '\0';
                sf = SYS_STRRCHR(compPath, '/');
                if (sf) sf++;
                sb = SYS_STRRCHR(compPath, '\\');
                if (sb) sb++;
                compFile = (sf && sb) ? ((sf > sb) ? sf : sb) : (sf ? sf :
                               (sb ? sb : compPath));
            }
            prev_start_index = 0;

#ifdef FS_MEM_DBG
            STATE.memdbgid = "fntset->cfnt";
#endif
            fntset->cfnt = FSS_calloc(_PS_ ltt->numComponents * sizeof(CFNT));
            if (!fntset->cfnt)
            {
                if ((FNTSET_OT_TABLE *)(fntset->gpos))
                    FSS_free(_PS_ (FNTSET_OT_TABLE *)(fntset->gpos));
                if ((FNTSET_OT_TABLE *)(fntset->gsub))
                    FSS_free(_PS_ (FNTSET_OT_TABLE *)(fntset->gsub));
                if ((FNTSET_OT_TABLE *)(fntset->gdef))
                    FSS_free(_PS_ (FNTSET_OT_TABLE *)(fntset->gdef));
                if ((FNTSET_OT_TABLE *)(fntset->cmap))
                    FSS_free(_PS_ (FNTSET_OT_TABLE *)(fntset->cmap));
                FSS_free(_PS_ (FILECHAR *)(fntset->name));
                FsLtt_done(ltt);
                FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, ltt);
                if (fntset->path) FSS_free(_PS_ (FILECHAR *)(fntset->path));
                FSS_free(_PS_ fntset);
                FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, compPath);
                return NULL;
            }

            buffer = (FILECHAR *)FsScratchSpace_reserve(&STATE.server->scratch, FS_state_ptr,
                     256 * sizeof(FILECHAR));
            if (!buffer)
            {
                FSS_free(_PS_ (CFNT *)(fntset->cfnt));
                if ((FNTSET_OT_TABLE *)(fntset->gpos))
                    FSS_free(_PS_ (FNTSET_OT_TABLE *)(fntset->gpos));
                if ((FNTSET_OT_TABLE *)(fntset->gsub))
                    FSS_free(_PS_ (FNTSET_OT_TABLE *)(fntset->gsub));
                if ((FNTSET_OT_TABLE *)(fntset->gdef))
                    FSS_free(_PS_ (FNTSET_OT_TABLE *)(fntset->gdef));
                if ((FNTSET_OT_TABLE *)(fntset->cmap))
                    FSS_free(_PS_ (FNTSET_OT_TABLE *)(fntset->cmap));
                FSS_free(_PS_ (FILECHAR *)(fntset->name));
                FsLtt_done(ltt);
                FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, ltt);
                if (fntset->path) FSS_free(_PS_ (FILECHAR *)(fntset->path));
                FSS_free(_PS_ fntset);
                FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, compPath);
                return NULL;
            }
            for (cfnt = fntset->cfnt, cnt = 0; cnt < ltt->numComponents; cfnt++, cnt++)
            {
                FsLttComponent *comp;
                FILECHAR *file;
                FS_BYTE *address;
                FS_ULONG offset;
                FS_ULONG length;
                FS_ULONG result;

                comp = ltt->components + cnt;

                file = NULL;
                address = NULL;
                offset = 0;
                length = 0;

                if (comp->targetCId)
                {
                    void *cb_address;

                    cb_address = NULL;
                    buffer[0] = '\0';

                    result = FS_callback_get_target(comp->targetCId,
                                                    &cb_address,
                                                    buffer, 256,
                                                    &offset,
                                                    &length);
                    if ((result != SUCCESS) ||
                            ((cb_address == NULL) && (buffer[0] == '\0')))
                    {
                        FS_USHORT ncmp;

                        for (ncmp = 0, cfnt = (CFNT *)(fntset->cfnt); ncmp < cnt; ncmp++, cfnt++)
                        {
                            LFNT *lfnt;

                            lfnt = (LFNT *)(cfnt->lfnt);

                            lfnt->fntset_refs--;

                            if (lfnt->fntset_refs < 1)
                            {
                                delete_lfnt(_PS_ lfnt);
                            }
                            cfnt->name = 0;
                        }
                        FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, buffer);
                        FSS_free(_PS_ (CFNT *)(fntset->cfnt));
                        if ((FNTSET_OT_TABLE *)(fntset->gpos))
                            FSS_free(_PS_ (FNTSET_OT_TABLE *)(fntset->gpos));
                        if ((FNTSET_OT_TABLE *)(fntset->gsub))
                            FSS_free(_PS_ (FNTSET_OT_TABLE *)(fntset->gsub));
                        if ((FNTSET_OT_TABLE *)(fntset->gdef))
                            FSS_free(_PS_ (FNTSET_OT_TABLE *)(fntset->gdef));
                        if ((FNTSET_OT_TABLE *)(fntset->cmap))
                            FSS_free(_PS_ (FNTSET_OT_TABLE *)(fntset->cmap));
                        FSS_free(_PS_ (FILECHAR *)(fntset->name));
                        FsLtt_done(ltt);
                        FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, ltt);
                        if (fntset->path) FSS_free(_PS_ (FILECHAR *)(fntset->path));
                        FSS_free(_PS_ fntset);
                        FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, compPath);

                        STATE.error = result ? result : ERR_INVALID_ADDRESS;

                        return NULL;
                    }
                    else if (cb_address)
                    {
                        address = cb_address;
                    }
                    else
                    {
                        file = buffer;
                    }
                }
                else if (comp->targetDir)
                {
                    if (!SYS_STRCMP(comp->targetDir, "."))
                    {
                        FS_ULONG len = (FS_ULONG)(255 - (compFile - compPath));
                        SYS_STRNCPY(compFile, comp->fileName, len);
                        compPath[255] = '\0';
                        file = compPath;
                    }
                    else if (!SYS_STRNCMP(comp->targetDir, "./", 2) ||
                             !SYS_STRNCMP(comp->targetDir, ".\\", 2) ||
                             !SYS_STRNCMP(comp->targetDir, "../", 3) ||
                             !SYS_STRNCMP(comp->targetDir, "..\\", 3))
                    {
                        FS_ULONG len = (FS_ULONG)(255 - (compFile - compPath));
                        FS_ULONG wid = SYS_STRLEN(comp->targetDir);
                        SYS_STRNCPY(compFile, comp->targetDir, len);
                        len -= wid;
                        if ((comp->targetDir[wid - 1] != '/') &&
                                (comp->targetDir[wid - 1] != '\\'))
                        {
                            if (len >= 1)
                            {
                                compFile[wid] = '/';
                                wid += 1;
                                len -= 1;
                            }
                            compFile[wid] = '\0';
                        }
                        SYS_STRNCPY(compFile + wid, comp->fileName, len);
                        compPath[255] = '\0';
                        file = compPath;
                    }
                    else
                    {
                        FS_ULONG len;

                        SYS_STRNCPY(buffer, comp->targetDir, 255);
                        buffer[255] = '\0';

                        len = SYS_STRLEN(buffer);

                        if (len &&
                            (buffer[len - 1] != '/') &&
                            (buffer[len - 1] != '\\'))
                        {
                            if (len < 254)
                                SYS_STRNCAT(buffer, "/", 1);
                            len++;
                        }
                        if ((len + SYS_STRLEN(comp->fileName)) < 256)
                        {
                            SYS_STRNCAT(buffer, comp->fileName, SYS_STRLEN(comp->fileName));
                        }
                        buffer[255] = '\0';
                        file = buffer;
                    }
                    offset = comp->targetOff;
                    length = comp->targetLen;
                }
                else if (comp->targetPtr)
                {
                    address = (FS_BYTE *)((long)(comp->targetPtr));
                }
                else    /* embedded */
                {
                    file = path;
                    address = memptr;
                    offset = comp->fontOffset;
                    length = comp->fontSize;
                }
                {
                    FS_BYTE ftype;

                    ftype = fontfile_type(_PS_ file, address, offset);
                    if (STATE.error != SUCCESS)
                    {
                        FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, buffer);
                        FSS_free(_PS_ (CFNT *)(fntset->cfnt));
                        if ((FNTSET_OT_TABLE *)(fntset->gpos))
                            FSS_free(_PS_ (FNTSET_OT_TABLE *)(fntset->gpos));
                        if ((FNTSET_OT_TABLE *)(fntset->gsub))
                            FSS_free(_PS_ (FNTSET_OT_TABLE *)(fntset->gsub));
                        if ((FNTSET_OT_TABLE *)(fntset->gdef))
                            FSS_free(_PS_ (FNTSET_OT_TABLE *)(fntset->gdef));
                        if ((FNTSET_OT_TABLE *)(fntset->cmap))
                            FSS_free(_PS_ (FNTSET_OT_TABLE *)(fntset->cmap));
                        FSS_free(_PS_ (FILECHAR *)(fntset->name));
                        FsLtt_done(ltt);
                        FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, ltt);
                        if (fntset->path) FSS_free(_PS_ (FILECHAR *)(fntset->path));
                        FSS_free(_PS_ fntset);
                        FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, compPath);
                        return NULL;
                    }

                    result = FS_SERVER_init_CFNT(_PS_ cfnt, ftype,
                                                 file, address,
                                                 comp->ttcIndex,
                                                 offset, length);
                }
                if (result != SUCCESS)
                {
                    FS_USHORT ncmp;

                    for (ncmp = 0, cfnt = (CFNT *)(fntset->cfnt); ncmp < cnt; ncmp++, cfnt++)
                    {
                        LFNT *lfnt;

                        lfnt = (LFNT *)(cfnt->lfnt);

                        lfnt->fntset_refs--;

                        if (lfnt->fntset_refs < 1)
                        {
                            delete_lfnt(_PS_ lfnt);
                        }
                        cfnt->name = 0;
                    }
                    FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, buffer);
                    FSS_free(_PS_ (CFNT *)(fntset->cfnt));
                    if ((FNTSET_OT_TABLE *)(fntset->gpos))
                        FSS_free(_PS_ (FNTSET_OT_TABLE *)(fntset->gpos));
                    if ((FNTSET_OT_TABLE *)(fntset->gsub))
                        FSS_free(_PS_ (FNTSET_OT_TABLE *)(fntset->gsub));
                    if ((FNTSET_OT_TABLE *)(fntset->gdef))
                        FSS_free(_PS_ (FNTSET_OT_TABLE *)(fntset->gdef));
                    if ((FNTSET_OT_TABLE *)(fntset->cmap))
                        FSS_free(_PS_ (FNTSET_OT_TABLE *)(fntset->cmap));
                    FSS_free(_PS_ (FILECHAR *)(fntset->name));
                    FsLtt_done(ltt);
                    FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, ltt);
                    if (fntset->path) FSS_free(_PS_ (FILECHAR *)(fntset->path));
                    FSS_free(_PS_ fntset);
                    FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, compPath);

                    STATE.error = result;

                    return NULL;
                }
                if (cnt == ltt->mtrxComponent)
                {
                    fntset->metric_font = cfnt->lfnt;
                }
                cfnt->bmp.range1 = comp->bmp.range1;
                cfnt->bmp.range2 = comp->bmp.range2;
                cfnt->smp.range1 = comp->smp.range1;
                cfnt->smp.range2 = comp->smp.range2;
                cfnt->sip.range1 = comp->sip.range1;
                cfnt->sip.range2 = comp->sip.range2;
                cfnt->ssp.range1 = comp->ssp.range1;
                cfnt->ssp.range2 = comp->ssp.range2;

                cfnt->adjOffset = comp->adjustmentsOffset;
                cfnt->numAdjust = comp->numAdjustments;

                /* if bold, convert boldPercent to fraction */
                if (comp->bold)
                {
                    cfnt->embolden = 1;
                    cfnt->bold_pct = FixMul(comp->boldPercent, 655);
                }
                /* if italic, convert italicAngle to radians which is approx. tan(italicAngle) */
                if (comp->italic)
                {
                    FS_FIXED a3;
                    cfnt->italicize = 1;
                    cfnt->italic_angle = FixMul(comp->italicAngle, 1144);
                    a3 = cfnt->italic_angle;
                    a3 = FixMul(a3, a3);
                    a3 = FixMul(a3, cfnt->italic_angle);
                    a3 = FixDiv(a3, 196608); /* a cubed divided by 3.0 */
                    cfnt->italic_angle += a3;
                }
                if (cnt > 0)
                {
                    cfnt->start_index = (comp - 1)->numGlyphs +
                                        (comp - 1)->numIcons +
                                        prev_start_index;
                }
                prev_start_index = cfnt->start_index;

#ifndef FS_LOAD_UPFRONT
                /* get lfnt/cfnt name from ltt */
                if(cfnt->lfnt->name == NULL)
                {
                    FILECHAR *dup;
#ifdef FS_MEM_DBG
                    STATE.memdbgid = "lfnt->name";
#endif
                    dup = FS_strdup(_PS_ comp->fontName);
                    if (dup)
                    {
                        cfnt->lfnt->name = dup;
                        cfnt->name = cfnt->lfnt->name;
                    }
                }
#endif /* FS_LOAD_UPFRONT */
            } /* for each cfnt */

            FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, buffer);

            fntset->ref_count = 0;
            fntset->count_added = 0;

            /* prepend to list of available font sets */
            fntset->next = STATE.server->font_sets;
            STATE.server->font_sets = fntset;

            FsLtt_done(ltt);
            FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, ltt);
            FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, compPath);

            return fntset;
        }
#else
        {
            lttPath = lttPath;
            FSS_free(_PS_ fntset->path);
            FSS_free(_PS_ fntset);
            STATE.error = ERR_LINKED_FONTS_UNDEF;
            return NULL;
        }
#endif /* FS_LINKED_FONTS */
    }
}

static FNTSET *
FS_SERVER_get_FNTSET(_DS_
                     FILECHAR *path,
                     FS_BYTE *memptr,
                     FS_ULONG index,
                     FS_ULONG data_offset,
                     FS_ULONG data_length,
                     FILECHAR *lttPath)
{
    FNTSET *fntset;

    fntset = FS_SERVER_find_FNTSET(_PS_ path, memptr,
                                   index, data_offset, data_length);
    if (fntset)
    {
        return fntset;
    }
    else
    {
        fntset = FS_SERVER_create_FNTSET(_PS_ path, memptr,
                                         index, data_offset, data_length,
                                         lttPath);
        if (fntset)
        {
            return fntset;
        }
        else
        {
            return NULL;
        }
    }
}

static FNTSET *
FS_STATE_find_named_FNTSET(_DS_ FILECHAR *name)
{
    ADDEDFNT *fnt;

    for (fnt = STATE.name_map; fnt; fnt = fnt->next)
    {
        if (FS_streq(fnt->name, name))
        {
            return fnt->set;
        }
    }
    return NULL;
}

/**
 * returns the component index of the metric font
 */
static FS_USHORT
FNTSET_metric_font_component(FNTSET *fntset)
{
    /* only doing this for multi-component font sets */
    if (fntset->num_fonts <= 1)
    {
        return 0;
    }
    else
    {
        CFNT *cfnt;
        FS_USHORT ncmp;

        for (ncmp = 0, cfnt = (CFNT *)(fntset->cfnt); ncmp < fntset->num_fonts; ncmp++, cfnt++)
        {
            if (cfnt->lfnt == fntset->metric_font)
            {
                return ncmp;
            }
        }
        return 0;
    }
}

static CFNT *
FNTSET_find_named_CFNT(FNTSET *fntset, FILECHAR *name)
{
    /* only doing this for multi-component font sets */
    if (fntset->num_fonts <= 1)
    {
        return NULL;
    }
    else
    {
        CFNT *cfnt;
        FS_USHORT ncmp;

        for (ncmp = 0, cfnt = (CFNT *)(fntset->cfnt); ncmp < fntset->num_fonts; ncmp++, cfnt++)
        {
            if (FS_streq(name, cfnt->name))
            {
                return cfnt;
            }
        }
        return NULL;
    }
}

static FNTSET *
FS_STATE_create_temp_FNTSET(_DS_ CFNT *cfnt)
{
    LFNT *lfnt;

    lfnt = (LFNT *)(cfnt->lfnt);

    /* there's a quicker way to do this, but... */

    return FS_SERVER_get_FNTSET(_PS_
                                (FILECHAR *)(lfnt->path),
                                lfnt->memptr,
                                lfnt->index,
                                lfnt->data_offset,
                                lfnt->data_length,
                                NULL);
    /* "count_added" not incremented */
}

/****************************************************************/
FS_LONG FSS_set_font(_DS_ FILECHAR *name)
{
    FNTSET *fntset;
    {
        FS_STATE *statep;

        statep = &STATE;

        do
        {
            fntset = FS_STATE_find_named_FNTSET(statep, name);
        }
        while (!fntset && ((statep = statep->parent) != NULL));
    }
    if (!fntset)
    {
        CFNT *cfnt;

        cfnt = NULL;

        /* one of the fonts in the current set ? */
        if (STATE.cur_typeset.capacity && STATE.cur_typeset.fntset)
        {
            cfnt = FNTSET_find_named_CFNT(STATE.cur_typeset.fntset, name);
        }
        if (!cfnt)
        {
            ADDEDFNT *fnt;

            /* one of the fonts in any of our known sets ? */

            for (fnt = STATE.name_map; fnt; fnt = fnt->next)
            {
                FNTSET *fs = fnt->set;

                /* no need to check this one again */
                if (fs == STATE.cur_typeset.fntset) continue;

                cfnt = FNTSET_find_named_CFNT(fs, name);
                if (cfnt) break;
            }
        }
        if (cfnt)
        {
            fntset = FS_STATE_create_temp_FNTSET(_PS_ cfnt);
            if (!fntset)
            {
                return STATE.error;
            }
        }
        else
        {
            return STATE.error = ERR_FONT_NOT_FOUND;
        }
    }
    /* fntset is valid now...could ASSERT(fntset), but no need */
    STATE.error = SUCCESS;

    /* already the current font set ? */
    if (STATE.cur_typeset.fntset == fntset)
    {
        return STATE.error = SUCCESS;
    }

    /* reset the current typeset */
    TYPESET_clear(_PS_ & STATE.cur_typeset);
    if (STATE.error) return STATE.error;

    TYPESET_init(_PS_ & STATE.cur_typeset, fntset);
    if (STATE.error) return STATE.error;

    STATE.cur_lfnt = NULL; /* lfnt not designated yet */
    STATE.cur_sfnt = NULL; /* sfnt not designated yet */

    return STATE.error;
}

static FILECHAR *
FS_STATE_unique_fntset_name(_DS_ FILECHAR *seed,
                            FILECHAR *name, FS_ULONG name_len)
{
    FS_CONST char prefix[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz";

    FS_LONG i;

    if (!name || (name_len < 2))
    {
        STATE.error = ERR_FONT_BUFFER_TOO_SMALL;
        return NULL;
    }

    SYS_STRNCPY(name, seed, name_len - 1);
    name[name_len - 1] = '\0';

    i = 0;

    while (FS_STATE_find_named_FNTSET(_PS_ name) && (i < 0x7FFFFFFFL))
    {
        /* name already used */

        /* create a unique name */

        FS_LONG j;
        FS_ULONG k;

        /* turn <i> into a base 52 prefix for <name> */
        j = i;
        k = 0;
        do
        {
            name[k++] = prefix[j % 52];
            j /= 52;
        }
        while ((j > 0) && (k < (name_len - 1)));

        if (k < (name_len - 1)) name[k++] = ' ';

        /* append all we can of <name> */
        SYS_STRNCPY(name + k, seed, name_len - 1 - k);

        i++;
    }
    name[name_len - 1] = '\0';
    return name;
}

/****************************************************************/
/* add to AVAILABLE_FONTS list */
FS_LONG FSS_add_font(_DS_ FILECHAR *name,
                     FILECHAR *path,
                     FS_BYTE *memptr,
                     FS_ULONG index)
{
    return FSS_add_font_with_offset(_PS_ name, path, memptr, index, 0, 0);
}

/****************************************************************/
/* add to AVAILABLE_FONTS list using data offset and length */
FS_LONG FSS_add_font_with_offset(_DS_ FILECHAR *name,
                                 FILECHAR *path,
                                 FS_BYTE *memptr,
                                 FS_ULONG index,
                                 FS_ULONG data_offset,
                                 FS_ULONG data_length)
{
    FNTSET* fntset;
    FILECHAR *lttPath;
    FILECHAR tempBuf[MAX_FONT_NAME_LEN];

    lttPath = path;

    if (memptr) path = NULL;

    fntset = FS_SERVER_get_FNTSET(_PS_ path, memptr,
                                  index, data_offset, data_length, lttPath);
    if (fntset)
    {
        /* is this FNTSET already known to client ? */

        ADDEDFNT *fnt;

        for (fnt = STATE.name_map; fnt; fnt = fnt->next)
        {
            if (fnt->set == fntset)
            {
                /* yes */

                /* is it the same name ? */

                /* legacy code could have NULL name (a la "itypetst") */

                if (!name || FS_streq(fnt->name, name))
                {
                    /* yes */

                    fnt->add_count++;

                    return STATE.error = SUCCESS;
                }
                else
                {
                    /* no */

                    return STATE.error = ERR_FONT_NAME_NOT_UNIQUE;
                }
            }
        }
        /* no, add this font to our list with given name */

        /* is the given name already used ? */

        /* legacy code could have NULL name (a la "itypetst") */

        if (name && FS_STATE_find_named_FNTSET(_PS_ name))
        {
            /* yes */

            return STATE.error = ERR_FONT_NAME_IN_USE;
        }
        else
        {
            /* no */

            /* legacy code could have NULL name (a la "itypetst") */

            if (!name)
            {
                name = tempBuf;

                FS_STATE_unique_fntset_name(_PS_ fntset->name,
                                            name, MAX_FONT_NAME_LEN);
            }

            if (FS_STATE_add_named_FNTSET(_PS_ name, fntset) == SUCCESS)
            {
                return STATE.error = SUCCESS;
            }
            else
            {
                return STATE.error;
            }
        }
    }
    else
    {
        return STATE.error;
    }
}

/********************************************************************************/
/********** extract the font name from path/memptr ******************************/
FS_LONG FSS_font_name(_DS_
                      FILECHAR *path,
                      FS_BYTE *memptr,
                      FS_ULONG index, FS_ULONG name_len, FILECHAR *name)
{
    return FSS_font_name_with_offset(_PS_ path, memptr, index,
                                     name_len, name, 0, 0);
}

/********************************************************************************/
/********** extract the font name from path/memptr ******************************/
FS_LONG FSS_font_name_with_offset(_DS_
                                  FILECHAR *path,
                                  FS_BYTE *memptr,
                                  FS_ULONG index,
                                  FS_ULONG name_len, FILECHAR *name,
                                  FS_ULONG data_offset,
                                  FS_ULONG data_length)
{
    FNTSET *fntset;
    FILECHAR *lttPath;

    lttPath = path;

    if (memptr) path = NULL;

    fntset = FS_SERVER_get_FNTSET(_PS_ path, memptr,
                                  index, data_offset, data_length, lttPath);
    if (fntset)
    {
        /* is this FNTSET already known to client ? */

        ADDEDFNT *fnt;

        for (fnt = STATE.name_map; fnt; fnt = fnt->next)
        {
            if (fnt->set == fntset)
            {
                /* yes */

                if ((SYS_STRLEN(fnt->name)) >= name_len)
                {
                    return STATE.error = ERR_FONT_BUFFER_TOO_SMALL;
                }
                else
                {
                    SYS_STRNCPY(name, fnt->name, name_len - 1);
                    name[name_len - 1] = '\0';

                    return STATE.error = SUCCESS;
                }
            }
        }
        /* no, need a unique name (assuming user will call FS_add_font next) */

        FS_STATE_unique_fntset_name(_PS_ fntset->name, name, name_len);

        return STATE.error = SUCCESS;
    }
    else
    {
        return STATE.error;
    }
}

/****************************************************************/
static FS_BOOLEAN any_sfnts_dependent_on_unshared_lfnts(_DS_ FNTSET *fntset)
{
    CFNT *cfnt;
    FS_USHORT ncmp;

    for (cfnt = (CFNT *)(fntset->cfnt), ncmp = 0; ncmp < fntset->num_fonts; ncmp++, cfnt++)
    {
        LFNT *lfnt;
        SFNT *sfnt;

        lfnt = (LFNT *)(cfnt->lfnt);

        if (lfnt->fntset_refs < 2)  /* less than 2 == 1 -> lfnt not shared */
        {
            for (sfnt = (SFNT *)(STATE.server->scaled_fonts); sfnt; sfnt = (SFNT *)(sfnt->next))
            {
                if (((LFNT *)(sfnt->lfnt) == lfnt) && sfnt->active_count)
                {
                    return 1;
                }
            }
        }
    }
    return 0;
}

/****************************************************************/
FS_LONG FSS_delete_font(_DS_ FILECHAR *name)
{
    FNTSET *fntset;
    ADDEDFNT *fnt, *prevfnt;

    fntset = NULL;
    prevfnt = NULL;
    for (fnt = STATE.name_map; fnt; fnt = fnt->next)
    {
        if (FS_streq(fnt->name, name))
        {
            fntset = fnt->set;
            break;
        }
        prevfnt = fnt;
    }
    if (!fnt || !fntset)
    {
        return STATE.error = ERR_FONT_NOT_FOUND;
    }
    if (fnt->add_count < 1)     /* this should never happen */
    {
        return STATE.error = ERR_DELETE_FONT;
    }
    else if (fnt->add_count > 1)
    {
        fnt->add_count--;

        /* this font was added to this client/STATE more than once */

        return STATE.error = SUCCESS;
    }

    /* at this point (fnt->add_count == 1) ...  */
    /* this client/STATE is done with this font */

    if (fntset->count_added < 1)       /* this should never happen */
    {
        return STATE.error = ERR_DELETE_FONT;
    }

    /* fntset->count_added >= 1 */

    if (fntset->count_added == 1)
    {
        /* we are the only client/STATE that added/loaded this font set */

        /* any sfnts that are dependent on our lfnts should stop us from
           removing this font set.
           but, each lfnt could be shared with another font set (in this
           thread/process or another one).
           if we assume that this is a well-behaved application, then
           we could assume that any sfnts dependent on SHARED lfnts are
           another font set's responsibility.
           so, only stop from deleting this font set if an sfnt is dependent
           on an lfnt that is NOT shared -- proof of a not-so-well behaved
           application.
        */
        if (any_sfnts_dependent_on_unshared_lfnts(_PS_ fntset))
        {
            return STATE.error = ERR_FONT_IN_USE;
        }
        /* same logic here as with sfnts...
           if a table ptr is associated directly with the font set, this
           stops our delete as it shows a not-so-well behaved application.
           if a table ptr is associated with a non-shared lfnt, again, stop
           our delete.
           but, if a table ptr is associated with a shared lfnt, assume this
           application is well behaved and the table ptr is not this font set's
           responsibility and we can go ahead and delete this font set.
        */
        if (any_used_fntset_or_unshared_lfnt_table_ptrs(_PS_ fntset))
        {
            return STATE.error = ERR_FONT_IN_USE;
        }
    }
    else /* fntset->count_added > 1 */
    {
        /* someone else besides us also added/loaded this font set */

        /* assume we are a well-behaved application and that if we were
           to find ANY sfnts dependent on our lfnts, that they are
           the responsibility of someone else.
           with this assumption, we don't even check for dependent sfnts.
        */
        /* same for the table ptrs...
           assume we are a well-behaved application and that if we were
           to find ANY used table ptrs (associated directly with the font set
           or with any of our lfnts), that they are the responsibility of
           someone else.
           with this assumption, we don't even check for used table ptrs.
        */
    }

    /* fntset->count_added >= 1 */

    {
        /* are we deleting the current font set?  */
        /* "a feature included by popular demand" */

        if (STATE.cur_typeset.capacity && STATE.cur_typeset.fntset == fntset)
        {
            if (fntset->ref_count < 1)
            {
                /* we have this font set as current fontset.
                   so, how could the ref_count be less than 1.

                   IT SHOULDN'T HAPPEN !!
                */
                return STATE.error = ERR_DELETE_FONT;
            }

            /* clear typeset and reduce fntset->ref_count by one */
            TYPESET_clear(_PS_ & STATE.cur_typeset);
            STATE.cur_lfnt = NULL;
            STATE.cur_sfnt = NULL;

            STATE.platform = 0xFFFF;
            STATE.encoding = 0xFFFF;
        }
        /* decrement the fontset count, as we will no longer know of it
         */
        fntset->count_added--;      /* *** DECREMENTING *** */

        /* remove our record of adding this font
         */
        fnt->add_count--;

        FSS_free(_PS_ fnt->name);
        if (prevfnt)
        {
            prevfnt->next = fnt->next;
        }
        else
        {
            STATE.name_map = fnt->next;
        }
        FSS_free(_PS_ fnt);


        /* squeeze the unused table ptrs
        */
        squeeze_table_ptrs(_PS_ fntset);
    }

    /* is this fontset a current fontset of any other client/STATE ?
     */
    if (fntset->ref_count > 0)
    {
        /* another client/STATE has this as current, so we cannot go on */

        return STATE.error = SUCCESS;
    }

    /* fntset->count_added >= 0 */

    if (fntset->count_added == 0)
    {
        kill_fntset(_PS_ fntset);

        return STATE.error = SUCCESS;
    }
    else /* fntset->count_added > 0 */
    {
        /* and, lastly, we can also squeeze the sfnt list.  that is,
           remove the sfnts that are dependent upon our lfnts, but are NOT
           active and do NOT have anything in cache.
           similarly, this should catch those cases where the remnants are
           due to our use and good behavior (freeing chars after use).
        */
        /* SHOULD WE ?
           do we also need to call trim_cache() to make this useful?
           should we kill all unused sfnts? or just the ones associated
           with this font set and its lfnts?
        trim_cache(_PS0_);
        kill_unused_sfnts(_PS0_);
        */

        /* the real goal is to close the FS_FILE pointers associated with the
           font files.  but, someone else is using this font set so this
           isn't possible, right?
           would it be wise to call unload_fnt->unload_ttf which closes
           the FS_FILE pointer?   this would force a reload by the others
        */

        /* nah.  to close the file pointers would allow someone to
           remove the flash memory where the font resides, messing the
           other user of the font up!

           let's let get_some_back() do its work if necessary.  try not
           to be too presumptuous.
        */
        return STATE.error = SUCCESS;
    }
}

/****************************************************************/
FS_LONG FSS_load_font(_DS_
                      FILECHAR *path,
                      FS_BYTE *memptr,
                      FS_ULONG index, FS_ULONG name_len, FILECHAR *name)
{
    return FSS_load_font_with_offset(_PS_ path, memptr, index,
                                     name_len, name, 0, 0);
}

/****************************************************************/
FS_LONG FSS_load_font_with_offset(_DS_
                                  FILECHAR *path,
                                  FS_BYTE *memptr,
                                  FS_ULONG index,
                                  FS_ULONG name_len, FILECHAR *name,
                                  FS_ULONG data_offset,
                                  FS_ULONG data_length)
{
    FNTSET *fntset;
    FILECHAR *lttPath;

    lttPath = path;

    if (memptr) path = NULL;

    fntset = FS_SERVER_get_FNTSET(_PS_ path, memptr,
                                  index, data_offset, data_length, lttPath);
    if (fntset)
    {
        /* is this FNTSET already known to client ? */

        ADDEDFNT *fnt;

        for (fnt = STATE.name_map; fnt; fnt = fnt->next)
        {
            if (fnt->set == fntset)
            {
                /* yes */

                if ((SYS_STRLEN(fnt->name) + 1) > name_len)
                {
                    return STATE.error = ERR_FONT_BUFFER_TOO_SMALL;
                }
                else
                {
                    fnt->add_count++;

                    SYS_STRNCPY(name, fnt->name, name_len - 1);
                    name[name_len - 1] = '\0';

                    return STATE.error = SUCCESS;
                }
            }
        }
        /* no, need a unique name with which to add this font to our list */

        FS_STATE_unique_fntset_name(_PS_ fntset->name, name, name_len);

        if (FS_STATE_add_named_FNTSET(_PS_ name, fntset) == SUCCESS)
        {
            return STATE.error = SUCCESS;
        }
        else
        {
            return STATE.error;
        }
    }
    else
    {
        return STATE.error;
    }
}

/****************************************************************/
/* returns structure describing a linked font                   */
FS_LONG FSS_get_link_info(_DS_ FILECHAR *path, FS_BYTE *memptr, FsLtt *info)
{
#ifdef FS_LINKED_FONTS
    FS_BYTE file_type;
    FS_FILE *fp;
    STATE.error = SUCCESS;

    if (memptr) path = NULL;

    /* IMPORTANT: this function should be eliminated.  Its value is
       questionable.  It does not utilize data_offset.  It ignores any
       name mapping that may occur.
    */

    file_type = fontfile_type(_PS_ path, memptr, 0);

    if (file_type != LTT_TYPE)
        return STATE.error = ERR_NOT_A_LTT;

    if (info == NULL)
        return STATE.error = ERR_NOT_A_LTT;

    FsLtt_init(_PS_ info);

    if (memptr)
    {
        FsLtt_load(info, memptr, 0);
    }
    else if (path)
    {
        fp = FS_open(_PS_ path);
        if (!fp)
        {
            STATE.error = ERR_FILE_OPEN;
            return STATE.error;
        }
        FsLtt_read(info, fp, 0);
        FS_close(_PS_ fp);
    }
    if (STATE.error)
        FsLtt_done(info); /* free ltt data */
#else
    info = info;
    memptr = memptr;
    path = path;
    STATE.error = ERR_LINKED_FONTS_UNDEF;
#endif /* FS_LINKED_FONTS */
    return STATE.error;
}

FS_LONG FSS_get_font_path(_DS_ FILECHAR *name,
                          FS_BYTE **memptr_p,
                          FILECHAR *path_buffer, FS_ULONG buffer_length,
                          FS_ULONG *index_p,
                          FS_ULONG *offset_p,
                          FS_ULONG *length_p)
{
    FNTSET *fntset;
    {
        FS_STATE *statep;

        statep = &STATE;

        do
        {
            fntset = FS_STATE_find_named_FNTSET(statep, name);
        }
        while (!fntset && ((statep = statep->parent) != NULL));
    }
    if (!fntset)
    {
        CFNT *cfnt;

        cfnt = NULL;

        /* one of the fonts in the current set ? */
        if (STATE.cur_typeset.capacity && STATE.cur_typeset.fntset)
        {
            cfnt = FNTSET_find_named_CFNT(STATE.cur_typeset.fntset, name);
        }
        if (!cfnt)
        {
            ADDEDFNT *fnt;

            /* one of the fonts in any of our known sets ? */

            for (fnt = STATE.name_map; fnt; fnt = fnt->next)
            {
                FNTSET *fs = fnt->set;

                /* no need to check this one again */
                if (fs == STATE.cur_typeset.fntset)
                    continue;

                cfnt = FNTSET_find_named_CFNT(fs, name);
                if (cfnt)
                    break;
            }
        }
        if (cfnt)
        {
            fntset = FS_STATE_create_temp_FNTSET(_PS_ cfnt);
            if (!fntset)
            {
                return STATE.error;
            }
        }
        else
        {
            return STATE.error = ERR_FONT_NOT_FOUND;
        }
    }
    /* fntset is valid now...could ASSERT(fntset), but no need */
    {
        if (fntset->path)
        {
            if (SYS_STRLEN(fntset->path) >= buffer_length)
            {
                return STATE.error = ERR_FONT_BUFFER_TOO_SMALL;
            }
            else
            {
                SYS_STRNCPY(path_buffer, fntset->path, buffer_length - 1);
                path_buffer[buffer_length - 1] = '\0';
            }
        }
        *memptr_p = fntset->memptr;
        *offset_p = fntset->data_offset;
        *length_p = fntset->data_length;

        if (fntset->num_fonts == 1)
        {
            LFNT *lfnt;
            CFNT *cfnt = fntset->cfnt;

            lfnt = (LFNT *)(cfnt->lfnt);
            if (!lfnt)
            {
                return STATE.error = ERR_BAD_LFNT;
            }
            *index_p = lfnt->index;
        }
    }
    return STATE.error = SUCCESS;
}

/****************************************************************/
/* returns unique name of cfnt font used to render         */
/* use to render character id in the current font set           */
FS_LONG FSS_get_name(_DS_ FS_ULONG id, FS_ULONG name_len, FILECHAR *name)
{
    FS_USHORT index, len = 0;
    FILECHAR *lfntname;
    *name = 0;
    index = map_char(_PS_ id, 0); /* set current font set and cur_font */
    if (STATE.error || index == 0)
        return STATE.error;

    lfntname = (FILECHAR *)(STATE.cur_typeset.tfntarray[STATE.cur_font].cfnt->name);
    if (lfntname)
        len = (FS_USHORT)SYS_STRLEN(lfntname);
    else
        return STATE.error = ERR_BAD_LFNT;
    if (len >= name_len)
        return STATE.error = ERR_FONT_BUFFER_TOO_SMALL;

    SYS_MEMCPY(name, lfntname, len);
    name[len] = 0;

    return STATE.error = SUCCESS;
}

/****************************************************************/
static FS_LONG
ttf_font_metrics(_DS_ LFNT *lfnt, FONT_METRICS *fm, FS_ULONG ttcIndex)
{
    TTF *ttf;
    TTF_CMAP *pcmap;
    CMAP_TAB *tabs;
    int i;

    ttf = (TTF *)(lfnt->fnt);
    if (!ttf)
    {
        return STATE.error = ERR_BAD_LFNT;
    }
    if (!ttf->memptr || (FS_VOID *)(ttf->decomp))
    {
        pcmap = ttf->cmap;
    }
    else
    {
        pcmap = ttf->cmap;
    }

    {
        TTF_MAXP maxp;
        if (FSS_get_table_structure(_PS_ TAG_maxp, &maxp) != SUCCESS)
            return STATE.error;
        /* Extract numGlyphs */
        fm->numGlyphs = maxp.numGlyphs;
    }

    {
        TTF_HEAD head;
        if (FSS_get_table_structure(_PS_ TAG_head, &head) != SUCCESS)
            return STATE.error;
        /* Extract "unitsPerEm", "font_bbox" from "head" table. */
        fm->unitsPerEm = head.unitsPerEm;
        fm->head_macStyle = head.macStyle;
        fm->metricsResolution = head.unitsPerEm;
        fm->font_bbox.xMax = head.xMax;
        fm->font_bbox.xMin = head.xMin;
        fm->font_bbox.yMax = head.yMax;
        fm->font_bbox.yMin = head.yMin;
    }
    {
        TTF_HHEA hhea;
        if (FSS_get_table_structure(_PS_ TAG_hhea, &hhea) != SUCCESS)
            return STATE.error;
        /* Extract "ascent", "descent" and "leading" from "hhea" table */
        fm->hhea_ascent  = hhea.yAscender;
        fm->hhea_descent = hhea.yDescender;
        fm->hhea_leading = hhea.yLineGap;
    }
    {
        TTF_OS2  os2;
        if (FSS_get_table_structure(_PS_ TAG_OS2, &os2) != SUCCESS)
        {
            if (STATE.error != ERR_TABLE_NOT_FOUND)
                return STATE.error;
        }
        else
        {
            /* Extract "ascent", "descent", "leading" and "embedding_bits"
            * from "os/2" table
            */
            fm->os2_win_ascent = os2.usWinAscent;
            fm->os2_win_descent = os2.usWinDescent;
            fm->os2_ascent = os2.sTypoAscender;
            fm->os2_descent = os2.sTypoDescender;
            fm->os2_leading = os2.sTypoLineGap;
            fm->embedding_bits = os2.fsType;
            fm->os2_fsSelection = os2.fsSelection;
        }
    }
    /* Extract encoding table data from "cmap" table. */
    tabs = (CMAP_TAB *) pcmap->tables;
    fm->num_cmap_tables = SWAPW(pcmap->number);
    if (fm->num_cmap_tables > MAX_MAPPINGS_LEN)
        fm->num_cmap_tables = MAX_MAPPINGS_LEN;
    for (i = 0; i < fm->num_cmap_tables; i++)
    {
        fm->mappings[i].platform = SWAPW(tabs[i].platform);
        fm->mappings[i].encoding = SWAPW(tabs[i].encoding);
    }

    {
        TTF_NAME *name;
        name = (TTF_NAME *)FSS_malloc(_PS_ sizeof(TTF_NAME));
        if (name)
        {
            if (FSS_get_table_structure(_PS_ TAG_name, name) != SUCCESS)
            {
                FSS_free(_PS_ name);
                return STATE.error;
            }
            /* Extract font-related name data from "name" table */
            SYS_STRNCPY(fm->font_name, name->font_name, MAX_FONT_NAME_LEN - 1);
            fm->font_name[MAX_FONT_NAME_LEN - 1] = '\0';
            SYS_STRNCPY(fm->font_family_name, name->font_family_name, MAX_FONT_FAMILY_NAME_LEN - 1);
            fm->font_family_name[MAX_FONT_FAMILY_NAME_LEN - 1] = '\0';
            SYS_STRNCPY(fm->copyright, name->copyright, MAX_COPYRIGHT_LEN - 1);
            fm->copyright[MAX_COPYRIGHT_LEN - 1] = '\0';
            FSS_free(_PS_ name);
        }
        else
        {
            fm->font_name[0] = '\0';
            fm->font_family_name[0] = '\0';
            fm->copyright[0] = '\0';
        }
    }

    if (!lfnt->fnt && (load_fnt(_PS_ lfnt) != SUCCESS))
        return STATE.error;

    /* Determine type of font */
    if (lfnt->fontflags & FONTFLAG_ACT3)
        fm->font_type |= FM_FLAGS_ACT3;
    if ((((TTF *)(lfnt->fnt))->ttc_header))
    {
        fm->font_type |= FM_FLAGS_TTC;
        fm->numFontsInTTC = ((TTF *)lfnt->fnt)->ttc_header->numFonts;
        fm->indexInTTC = ttcIndex;
    }
    if ((lfnt->fontflags & (FONTFLAG_CCC | FONTFLAG_DDD)))
        fm->font_type |= FM_FLAGS_CCC;
    if (lfnt->fontflags & FONTFLAG_STIK)
        fm->font_type |= FM_FLAGS_STIK;
    else
    {
        if (lfnt->fnt_type == CFF_TYPE)
            fm->font_type |= FM_FLAGS_CFF;
        else
            fm->font_type |= FM_FLAGS_TTF;
    }

#ifdef FS_ICONS
    if (ttf && ttf->icon_offset)
    {
        FS_ULONG num, offset;

        num = 0; /* coverity UNINIT_USE */

        offset = ttf->icon_offset + 8;
        ttf_read_buf(_PS_ ttf, offset, 4, (FS_BYTE *)&num);
        fm->numIcons = SWAPL(num);
    }
    else
        fm->numIcons = 0;
#else
    fm->numIcons = 0;
#endif

    return STATE.error = SUCCESS;
}

/****************************************************************/
FS_LONG FSS_get_glyph_metrics(_DS_ FS_USHORT index,
                              FS_SHORT *lsb, FS_SHORT *aw,
                              FS_SHORT *tsb, FS_SHORT *ah )
{
    *lsb = *aw = *tsb = *ah = 0;
#ifdef FS_RENDER
    if ( STATE.cur_sfnt )
    {
        if (check_sfnt(_PS0_)) /* required since senv may have been gutted */
            return STATE.error;

        if ( STATE.cur_sfnt->senv )
        {
            fsg_SplineKey *key = (fsg_SplineKey *)STATE.cur_sfnt->senv->ttkey;
            if ( key )
            {
                fsg_Metrics metrics;
                FS_USHORT start_index = STATE.cur_typeset.tfntarray[STATE.cur_font].cfnt->start_index;

                if ((index - start_index) < ((TTF_MAXP *)(key->maxp))->numGlyphs)
                {
                    get_glyph_metrics(_PS_ key, index - start_index, &metrics);
                    *aw  = metrics.aw;
                    *lsb = metrics.lsb;
                    *ah  = metrics.ah;
                    *tsb = metrics.tsb;
                }
            }
        }
    }
    else
        STATE.error = ERR_NO_CURRENT_SFNT;
#else
    index = index;
#endif
    return STATE.error;
}

/****************************************************************/
FS_LONG FSS_font_metrics(_DS_ FONT_METRICS *fm)
{
    FNTSET *fntset = STATE.cur_typeset.fntset;
    LFNT *lfnt;

    /* zero the font metrics structure */
    SYS_MEMSET(fm, 0, sizeof(FONT_METRICS));

    /* gotta be a current font set */
    if (!fntset)
    {
        return STATE.error = ERR_NO_CURRENT_FNTSET;
    }
    lfnt = (LFNT *)(fntset->metric_font);

    /* lfnt->fnt must exist as well */
    if (!lfnt->fnt && (load_fnt(_PS_ lfnt) != SUCCESS))
    {
        return STATE.error;
    }

    if (lfnt->fnt_type == TTF_TYPE || lfnt->fnt_type == CFF_TYPE)
    {
        return ttf_font_metrics(_PS_ lfnt, fm, 0);
    }
    else if (lfnt->fnt_type == TTC_TYPE)
    {
        return ttf_font_metrics(_PS_ lfnt, fm, lfnt->index);
    }
    else
    {
#ifdef FS_PFRR
        return pfr_font_metrics(_PS_ lfnt, fm);
#else
        return STATE.error = ERR_PFR_UNDEF;
#endif
    }
}

/****************************************************************/
static void
TTF_calc_ascender_descender_leading(TTF_OS2 *os2,
                                    TTF_HHEA *hhea,
                                    TTF_HEAD *head,
                                    FS_FIXED emSize,
                                    FS_FIXED *ascender,
                                    FS_FIXED *descender,
                                    FS_FIXED *leading,
                                    FsAscDescLeadSource *source)
{
    FS_SHORT sAscender = 0;
    FS_SHORT sDescender = 0;
    FS_SHORT sLeading;

    if (os2)
    {
        sAscender = (FS_SHORT)os2->usWinAscent;
        sDescender = (FS_SHORT)os2->usWinDescent;
    }

    /* To calculate ascender, descender, and leading we use the
       recommended formula for Windows/Mac compatibility found at
       http:/www.microsoft.com/OpenType/OTSpec/recom.htm
       in the section "Baseline to Baseline Distances"

       If OS/2 exists and usWinAscent and usWinDescent are not
       both zero then:

       sAscender  = usWinAscent
       sDescender = usWinDescent
       sLeading   = "external leading" =
                      MAX(0, LineGap - ((usWinAscent + usWinDescent) -
                          (hhea->ascender - hhea->descender)))

       If usWinAscent and usWinDescent are both 0 then we fall
       back to using the hhea metrics.
    */

    if (sAscender == 0 && sDescender == 0)
    {
        sAscender = hhea->yAscender;
        sDescender = -hhea->yDescender;
        sLeading = hhea->yLineGap;

        *source = FsAscDescLeadSource_TTF_HHEA;
    }
    else
    {
        sLeading = MAX(0, hhea->yLineGap - ((sAscender + sDescender) -
                          (hhea->yAscender - hhea->yDescender)));

        *source = FsAscDescLeadSource_WIN_MAC_COMPAT;
    }

    *ascender = LongMulDiv(sAscender, emSize, head->unitsPerEm);

    *descender = LongMulDiv(sDescender, emSize, head->unitsPerEm);

    *leading = LongMulDiv(sLeading, emSize, head->unitsPerEm);
}

/****************************************************************/
static FS_LONG
LFNT_get_table_structure(_DS_ LFNT *lfnt, FS_ULONG tag, FS_VOID *tableptr)
{
    TTF *ttf;

    if (lfnt->fnt_type == PFR_TYPE)
    {
        return STATE.error = ERR_TABLE_UNSUPPORTED;
    }

    /* lfnt->fnt must exist */
    if (!lfnt->fnt && (load_fnt(_PS_ lfnt) != SUCCESS))
    {
        return STATE.error;
    }
    ttf = (TTF *)(lfnt->fnt);

    switch (tag)
    {
    case TAG_head:
    {
        TTF_HEAD *phead;
        phead = (TTF_HEAD *)(ttf->head);
        SYS_MEMCPY(tableptr, phead, sizeof_TTF_HEAD_ );
        break;
    }
    case TAG_hhea:
    {
        TTF_HHEA *phhea;
        phhea = (TTF_HHEA *)(ttf->hhea);
        SYS_MEMCPY(tableptr, phhea, sizeof(TTF_HHEA) );
        break;
    }
    case TAG_vhea: /* optional table */
    {
        TTF_VHEA *pvhea;
        pvhea = (TTF_VHEA *)(ttf->vhea);
        if (!pvhea )
        {
            return STATE.error = ERR_TABLE_NOT_FOUND;
        }
        SYS_MEMCPY(tableptr, pvhea, sizeof(TTF_VHEA) );
        break;
    }
    case TAG_maxp:
    {
        TTF_MAXP *pmaxp;
        pmaxp = (TTF_MAXP *)(ttf->maxp);
        SYS_MEMCPY(tableptr, pmaxp, sizeof(TTF_MAXP) );
        break;
    }
    case TAG_OS2:
    {
        TTF_OS2  *pos2;
        pos2  = (TTF_OS2 *)(ttf->os2);
        if (!pos2)
            return STATE.error = ERR_TABLE_NOT_FOUND;
        SYS_MEMCPY(tableptr, pos2, sizeof(TTF_OS2) );
        break;
    }
    case TAG_name:
    {
        TTF_NAME *pname;
        pname = (TTF_NAME *)(ttf->name);
        SYS_MEMCPY(tableptr, pname, sizeof(TTF_NAME) );
        break;
    }
    default:
    {
        return STATE.error = ERR_TABLE_UNSUPPORTED;
    }
    }
    return STATE.error = SUCCESS;
}

/****************************************************************/
static FS_LONG
LFNT_get_vdmx_pair(_DS_ LFNT *lfnt,
                   FS_USHORT xppm, FS_USHORT yppm,
                   FS_SHORT *yMax, FS_SHORT *yMin)
{
    FS_ULONG offset;
    FS_USHORT group_offset = 0;
    FS_USHORT  num_ratios, recs;
    FS_BYTE startsz, endsz;
    sfnt_vdmxRatio ratio;
    FS_USHORT i;
    FS_BOOLEAN default_grouping;
    FS_BOOLEAN in_range;
    sfnt_vdmxTable vtable;
    TTF  *ttf;

    ttf = (TTF *)(lfnt->fnt);

    if (lfnt->fnt_type == PFR_TYPE)
        return STATE.error = ERR_TABLE_UNSUPPORTED;

    if ( !ttf->vdmx_offset )
        return STATE.error = ERR_TABLE_NOT_FOUND;

    num_ratios = 0;    /* without this line, coverity reports uninit_use */
    recs = 0;    /* without this line, coverity reports uninit_use */

    offset = ttf->vdmx_offset;
    offset += 4;
    ttf_read_buf(_PS_ ttf, offset, 2, (FS_BYTE *)&num_ratios);
    num_ratios = SWAPW(num_ratios);
    offset += 2;

    in_range = 0;

    /* finding right vdmx group for current ratio */
    for ( i = 0; i < num_ratios; i++ )
    {
        /* without following line, coverity reports uninit_use */
        ratio.bCharSet = ratio.xRatio = ratio.yStartRatio = ratio.yEndRatio = 0;

        ttf_read_buf(_PS_ ttf, offset + i * sizeof(sfnt_vdmxRatio),
                     sizeof(sfnt_vdmxRatio), (FS_BYTE *)&ratio);

        default_grouping = ( ratio.xRatio == 0) &&
                           ( ratio.yStartRatio == 0 ) &&
                           ( ratio.yEndRatio == 0 );

        if ( !default_grouping )
        {
            FS_BYTE adj1 = yppm * ratio.xRatio % ratio.yStartRatio;
            FS_BYTE adj2 = yppm * ratio.xRatio % ratio.yEndRatio;
            in_range = ( ( ratio.xRatio * yppm >= ratio.yStartRatio * xppm + adj1 ) &&
                         ( ratio.xRatio * yppm <= ratio.yEndRatio * xppm + adj2 ) );
        }

        if ( default_grouping || in_range )
        {
            offset += num_ratios * sizeof(sfnt_vdmxRatio) + i * sizeof(FS_USHORT);
            ttf_read_buf(_PS_ ttf, offset, 2, (FS_BYTE *)&group_offset);
            group_offset = SWAPW(group_offset);
            break;
        }
    }

    if ( i == num_ratios )
        return STATE.error = ERR_VDMX_RATIO;

    offset = ttf->vdmx_offset + group_offset;
    ttf_read_buf(_PS_ ttf, offset , 2, (FS_BYTE *)&recs);
    recs = SWAPW(recs);
    offset += 2;

    ttf_read_buf(_PS_ ttf, offset, 1, (FS_BYTE *)&startsz);
    offset += 1;

    ttf_read_buf(_PS_ ttf, offset, 1, (FS_BYTE *)&endsz);
    offset += 1;

    if ( (yppm < startsz) || (yppm > endsz) )
        return STATE.error = ERR_yPelHeight_NOT_FOUND;

    for ( i = 0; i < recs; i++ )
    {
        ttf_read_buf(_PS_ ttf, offset, sizeof(sfnt_vdmxTable), (FS_BYTE *)&vtable);
        vtable.yPelHeight = SWAPW(vtable.yPelHeight);
        vtable.yMax = SWAPW(vtable.yMax);
        vtable.yMin = SWAPW(vtable.yMin);
        if (vtable.yPelHeight == yppm )
        {
            *yMax = vtable.yMax;
            *yMin = vtable.yMin;
            break;
        }

        if (vtable.yPelHeight > yppm )
            return STATE.error = ERR_yPelHeight_NOT_FOUND; /* yPelHeight not in table */
        offset += sizeof(sfnt_vdmxTable);
    }

    if ( i == recs )
        return STATE.error = ERR_yPelHeight_NOT_FOUND; /* yPelHeight not in table */

    return STATE.error = SUCCESS;
}

/****************************************************************/
static FS_LONG
LFNT_get_ascender_descender_leading(_DS_ LFNT *lfnt,
                                    FS_FIXED emSize,
                                    FS_FIXED *ascender,
                                    FS_FIXED *descender,
                                    FS_FIXED *leading,
                                    FsAscDescLeadSource *source)
{
    if ((lfnt->fnt_type == TTF_TYPE) ||
        (lfnt->fnt_type == TTC_TYPE) ||
        (lfnt->fnt_type == CFF_TYPE) )
    {
        /* try vdmx table */
        {
            FS_SHORT yMax, yMin;
            TTF_HHEA hhea;
            LFNT_get_vdmx_pair(_PS_ lfnt,
                               FS_ROUND(emSize),
                               FS_ROUND(emSize), &yMax, &yMin);
            if (STATE.error == SUCCESS)
            {
                TTF_OS2 *os2;
                TTF_HEAD head;

                os2 = FSS_malloc(_PS_ sizeof(TTF_OS2));
                if (!os2) return STATE.error;

                *ascender = (FS_FIXED)yMax << 16;
                *descender = (FS_FIXED)(-yMin) << 16;

                if ((LFNT_get_table_structure(_PS_ lfnt, TAG_head, &head) != SUCCESS))
                {
                    FSS_free(_PS_ os2);
                    return STATE.error;
                }

                if (LFNT_get_table_structure(_PS_ lfnt, TAG_OS2,  os2) == SUCCESS)
                    *leading = LongMulDiv(os2->sTypoLineGap, emSize, head.unitsPerEm);
                else
                {
                    if (LFNT_get_table_structure(_PS_ lfnt, TAG_hhea, &hhea) != SUCCESS)
                    {
                        FSS_free(_PS_ os2);
                        return STATE.error;
                    }
                    else
                        *leading = LongMulDiv(hhea.yLineGap, emSize, head.unitsPerEm);
                }
                *source = FsAscDescLeadSource_VDMX_TABLE;
#ifdef FS_STIK
                /* correct stroke font values for half the current stroke width */
                if (lfnt->fontflags & FONTFLAG_STIK)
                {
                    FS_FIXED halfwidth = FixMul(emSize, STATE.stroke_pct) >> 1;
                    *ascender  += halfwidth;
                    *descender += halfwidth;
                }
#endif
                FSS_free(_PS_ os2);
                return SUCCESS;
            }
            else STATE.error = SUCCESS; /* reset */
        }
        /* no vdmx pair, move on to table values */
        {
            TTF_OS2 *os2;
            TTF_HHEA hhea;
            TTF_HEAD head;

            os2 = FSS_malloc(_PS_ sizeof(TTF_OS2));
            if (!os2) return STATE.error;

            if ((LFNT_get_table_structure(_PS_ lfnt, TAG_hhea, &hhea) != SUCCESS) ||
                    (LFNT_get_table_structure(_PS_ lfnt, TAG_head, &head) != SUCCESS))
            {
                FSS_free(_PS_ os2);
                return STATE.error;
            }

            if (LFNT_get_table_structure(_PS_ lfnt, TAG_OS2,  os2) == SUCCESS)
                TTF_calc_ascender_descender_leading(os2, &hhea, &head,
                                                    emSize,
                                                    ascender,
                                                    descender,
                                                    leading,
                                                    source);
            else
                TTF_calc_ascender_descender_leading(0, &hhea, &head,
                                                    emSize,
                                                    ascender,
                                                    descender,
                                                    leading,
                                                    source);

#ifdef FS_STIK
            /* correct stroke font values for half the current stroke width */
            if (lfnt->fontflags & FONTFLAG_STIK)
            {
                FS_FIXED halfwidth = FixMul(emSize, STATE.stroke_pct) >> 1;
                *ascender  += halfwidth;
                *descender += halfwidth;
            }
#endif
            FSS_free(_PS_ os2);
            return STATE.error = SUCCESS;
        }
    }
    else
    {
#ifdef FS_PFRR
        PFR *pfr;
        FS_SHORT sAscender;
        FS_SHORT sDescender;
        FS_SHORT sLeading;

        /* lfnt->fnt must exist */
        if (!lfnt->fnt && (load_fnt(_PS_ lfnt) != SUCCESS))
        {
            return STATE.error;
        }
        pfr = (PFR *)(lfnt->fnt);

        sAscender = pfr->ascent;
        sDescender = pfr->descent;
        sLeading = pfr->externalleading;

        *ascender = LongMulDiv(sAscender, emSize, pfr->metricsResolution);

        *descender = LongMulDiv(sDescender, emSize, pfr->metricsResolution);

        *leading = LongMulDiv(sLeading, emSize, pfr->metricsResolution);

        *source = FsAscDescLeadSource_MS_COMPAT;

        return STATE.error = SUCCESS;
#else
        return STATE.error = ERR_PFR_UNDEF;
#endif
    }
}

/****************************************************************/
#ifdef FS_GET_ASCENDER_DESCENDER_LEADING__USE_MERGED_TABLES
static FS_LONG
curr_FNTSET_get_ascender_descender_leading(_DS_ FS_FIXED emSize,
        FS_FIXED *ascender,
        FS_FIXED *descender,
        FS_FIXED *leading,
        FsAscDescLeadSource *source)
{
    TTF_OS2  os2;
    TTF_HHEA hhea;
    TTF_HEAD head;

    if ((FSS_get_table_structure(_PS_ TAG_hhea, &hhea) != SUCCESS) ||
        (FSS_get_table_structure(_PS_ TAG_head, &head) != SUCCESS))
    {
        return STATE.error;
    }
    if (FSS_get_table_structure(_PS_ TAG_OS2,  &os2)  == SUCCESS)
        TTF_calc_ascender_descender_leading(&os2, &hhea, &head,
                                            emSize,
                                            ascender, descender, leading,
                                            source);
    else
        TTF_calc_ascender_descender_leading(0, &hhea, &head,
                                            emSize,
                                            ascender, descender, leading,
                                            source);
    return STATE.error = SUCCESS;
}
#endif

/****************************************************************/
FS_LONG FSS_get_ascender_descender_leading(_DS_
        FS_FIXED *ascender,
        FS_FIXED *descender,
        FS_FIXED *leading,
        FsAscDescLeadSource *source)
{
    FNTSET *fntset;

    fntset = STATE.cur_typeset.fntset;
    if (!fntset)
    {
        return STATE.error = ERR_NO_CURRENT_FNTSET;
    }

    if ((STATE.scale[0] == 0) && (STATE.scale[1] == 0) &&
        (STATE.scale[2] == 0) && (STATE.scale[3] == 0))
    {
        *ascender = 0;
        *descender = 0;
        *leading = 0;
        *source = FsAscDescLeadSource_UNKNOWN;
        return STATE.error = SUCCESS;
    }

#ifdef FS_GET_ASCENDER_DESCENDER_LEADING__USE_MERGED_TABLES
    {
        FS_LONG errcode;
        FS_FIXED xppm, yppm, tan_s;

        /* rotationally invariant yppm comes from STATE.scale */

        errcode = get_scale_inputs(STATE.scale, &xppm, &yppm, &tan_s);
        if (errcode)
        {
            return STATE.error = errcode;
        }
        errcode = curr_FNTSET_get_ascender_descender_leading(_PS_ yppm,
                  ascender,
                  descender,
                  leading, source);
        return STATE.error = errcode;
    }
#else
    {
        TFNT *tfnt;
        FS_USHORT num_fonts;

        tfnt = STATE.cur_typeset.tfntarray;
        if (!tfnt)
        {
            return STATE.error = ERR_BAD_TYPESET;
        }
        *ascender = -(9999 << 16);
        *descender = -(9999 << 16);
        *leading = -(9999 << 16);
        *source = FsAscDescLeadSource_UNKNOWN;

        num_fonts = fntset->num_fonts;

        while (num_fonts--)
        {
            CFNT *cfnt = tfnt->cfnt;
            LFNT *lfnt;
            SFNT *sfnt = tfnt->sfnt;
            SENV *senv;
            FS_FIXED locascender;
            FS_FIXED locdescender;
            FS_FIXED locleading;
            FsAscDescLeadSource locsource;
            FS_USHORT countWins;

            if (!cfnt || !sfnt)
            {
                return STATE.error = ERR_NO_CURRENT_SFNT;
            }
            lfnt = (LFNT *)(cfnt->lfnt);

            /* restore lfnt, sfnt if needed */
            STATE.cur_lfnt = lfnt;
            STATE.cur_sfnt = sfnt;

            if (check_sfnt(_PS0_))
            {
                return STATE.error;
            }
            /* rotationally invariant yppm comes from SENV */
            senv = (SENV *)(sfnt->senv);
            if (!senv)
            {
                return STATE.error = ERR_BAD_SFNT;  /* should not happen */
            }
            LFNT_get_ascender_descender_leading(_PS_ lfnt,
                                                (FS_FIXED)senv->yppm << 16,
                                                &locascender,
                                                &locdescender,
                                                &locleading,
                                                &locsource);
            if (STATE.error)
            {
                return STATE.error;
            }
            if (sfnt->vertical_shift)
            {
                FS_FIXED shift = sfnt->vertical_shift << 16;

                locascender += shift;
                locdescender -= shift;
            }
            countWins = 0;

            if (locascender > *ascender)
            {
                *ascender = locascender;
                countWins++;
            }

            if (locdescender > *descender)
            {
                *descender = locdescender;
                countWins++;
            }

            if (locleading > *leading)
            {
                *leading = locleading;
                countWins++;
            }

            if ((*source == FsAscDescLeadSource_UNKNOWN) || (countWins == 3))
            {
                *source = locsource;
            }
            else if ((countWins > 0) && (*source != locsource))
            {
                *source = FsAscDescLeadSource_FONTSET_MIXED;
            }
            tfnt++;
        }
        return STATE.error = SUCCESS;
    }
#endif
}

/* more swapping macros */
#if (NEED_TO_SWAP)

/*#define SWAP_TTC_HEADER(p) swap_ttc_header(p)*/
/*#define SWAP_TTF_HEADER(p) swap_ttf_header(p)*/
#define SWAP_HEAD(p)       swap_head(p)
#define SWAP_HHEA(p)       swap_hhea(p)
#define SWAP_VHEA(p)       swap_vhea(p)
#define SWAP_MAXP(p)       swap_maxp(p)

#else

/*#define SWAP_TTC_HEADER(p)*/
/*#define SWAP_TTF_HEADER(p)*/
#define SWAP_HEAD(p)
#define SWAP_HHEA(p)
#define SWAP_VHEA(p)
#define SWAP_MAXP(p)

#endif

/****************************************************************/
static FS_LONG
FNTSET_get_table_structure(_DS_ FNTSET *fntset,
                           FS_ULONG tag, FS_VOID *tableptr)
{
    switch (tag)
    {
    case TAG_head:
        if (fntset->head_offset)
        {
            FS_ULONG len;
            FS_BYTE *raw;

            raw = FSS_get_table(_PS_ tag, TBL_EXTRACT, &len);
            if (!raw) return STATE.error;

            SYS_MEMCPY(tableptr, raw, sizeof_TTF_HEAD_);

            FSS_free_table(_PS_ raw);

            SWAP_HEAD(tableptr);

            return STATE.error = SUCCESS;
        }
        break;

    case TAG_hhea:
        if (fntset->hhea_offset)
        {
            FS_ULONG len;
            FS_BYTE *raw;

            raw = FSS_get_table(_PS_ tag, TBL_EXTRACT, &len);
            if (!raw) return STATE.error;

            SYS_MEMCPY(tableptr, raw, sizeof(TTF_HHEA));

            FSS_free_table(_PS_ raw);

            SWAP_HHEA(tableptr);

            return STATE.error = SUCCESS;
        }
        break;

    case TAG_vhea:
        if (fntset->vhea_offset)
        {
            FS_ULONG len;
            FS_BYTE *raw;

            raw = FSS_get_table(_PS_ tag, TBL_EXTRACT, &len);
            if (!raw) return STATE.error;

            SYS_MEMCPY(tableptr, raw, sizeof(TTF_VHEA));

            FSS_free_table(_PS_ raw);

            SWAP_VHEA(tableptr);

            return STATE.error = SUCCESS;
        }
        break;

    case TAG_maxp:
        if (fntset->maxp_offset)
        {
            FS_ULONG len;
            FS_BYTE *raw;

            raw = FSS_get_table(_PS_ tag, TBL_EXTRACT, &len);
            if (!raw) return STATE.error;

            SYS_MEMCPY(tableptr, raw, sizeof(TTF_MAXP));

            FSS_free_table(_PS_ raw);

            SWAP_MAXP(tableptr);

            return STATE.error = SUCCESS;
        }
        break;

    case TAG_OS2:
        if (fntset->os_2_offset)
        {
            FS_ULONG len;
            FS_BYTE *raw;

            raw = FSS_get_table(_PS_ tag, TBL_EXTRACT, &len);
            if (!raw) return STATE.error;

            get_os2(tableptr, raw, len);

            FSS_free_table(_PS_ raw);

            return STATE.error = SUCCESS;
        }
        break;

    case TAG_name:
        if (fntset->name_offset)
        {
            FS_ULONG len;
            FS_BYTE *raw;

            raw = FSS_get_table(_PS_ tag, TBL_EXTRACT, &len);
            if (!raw) return STATE.error;
            else
            {
                TTF_NAME *name;

                name = (TTF_NAME *)get_abbreviated_name(_PS_ raw);
                if (!name) return STATE.error;
                else
                {
                    SYS_MEMCPY(tableptr, name, sizeof(TTF_NAME));

                    FSS_free(_PS_ name);
                }
                FSS_free_table(_PS_ raw);

                return STATE.error = SUCCESS;
            }
        }
        break;

    default:
        return STATE.error = ERR_TABLE_UNSUPPORTED;
    }
    return STATE.error = ERR_TABLE_NOT_FOUND;
}

/****************************************************************/
FS_LONG FSS_get_table_structure(_DS_ FS_ULONG tag, FS_VOID *tableptr)
{
    FNTSET *fntset;
    FS_LONG errval;

    fntset = STATE.cur_typeset.fntset;
    if (!fntset)
    {
        return STATE.error = ERR_NO_CURRENT_FNTSET;
    }
    errval = FNTSET_get_table_structure(_PS_ fntset, tag, tableptr);
    if (errval == ERR_TABLE_NOT_FOUND)
    {
        /* table not merged in the fntset, so get from metric font */
        LFNT *lfnt;

        lfnt = (LFNT *)(fntset->metric_font);

        errval = LFNT_get_table_structure(_PS_ lfnt, tag, tableptr);
    }
    return errval;
}

#ifdef FS_HINTS

/****************************************************************/
/* query hdmx table for advance width of glyph at given xppm    */
/* xppm is used for non-square aspect ratios per OpenType spec  */
static FS_LONG SFNT_get_hdmx_width(_DS_ SFNT *sfnt, FS_ULONG index, FS_SHORT *width)
{
    TTF  *ttf;
    FS_ULONG i, offset, recSize;
    FS_USHORT numRecs, numGlyphs;
    FS_ULONG xppm;
    LFNT *lfnt;
    SENV *senv;

    *width = 0;

    if (sfnt == 0)    /* check the SFNT */
        return STATE.error = ERR_NO_CURRENT_SFNT;

    lfnt = (LFNT *)(sfnt->lfnt);
    if (lfnt == 0)  /* check the LFNT */
        return STATE.error = ERR_NO_CURRENT_LFNT;

    if (lfnt->fnt_type == PFR_TYPE)
        return STATE.error = ERR_TABLE_UNSUPPORTED;

    ttf = (TTF *)(lfnt->fnt);

    senv = (SENV *)(STATE.cur_sfnt->senv);
    if (senv == 0)
        return STATE.error = ERR_BAD_SFNT; /* should never happen */
    xppm = senv->xppm;

    if ( !ttf->hdmx_offset )
        return STATE.error = ERR_TABLE_NOT_FOUND;

    if (senv->hdmx_group_offset == 0) /* first time searching for xppm */
    {
        numRecs = 0; /* without this line, coverity reports uninit_use */
        recSize = 0; /* without this line, coverity reports uninit_use */

        offset = ttf->hdmx_offset;
        offset += 2;
        ttf_read_buf(_PS_ ttf, offset, 2, (FS_BYTE *)&numRecs);
        numRecs = SWAPW(numRecs);
        offset += 2;
        ttf_read_buf(_PS_ ttf, offset, 4, (FS_BYTE *)&recSize);
        recSize = SWAPL(recSize);
        offset += 4; /* points to start of device records array */

        numGlyphs = ttf->maxp->numGlyphs;
        if (index >= numGlyphs)
            return STATE.error = ERR_BAD_GLYF_INDEX;

        /* find right hdmx group for current ratio */
        for ( i = 0; i < numRecs; i++ )
        {
            FS_BYTE ppem;

            ttf_read_buf(_PS_ ttf, offset, 1, (FS_BYTE *)&ppem);
            if (ppem > xppm)
                break; /* record not found */

            if (ppem == xppm)
            {
                FS_BYTE w;
                ttf_read_buf(_PS_ ttf, offset + 2 + index, 1, (FS_BYTE *)&w);
                *width = (FS_SHORT)w;
                senv->hdmx_group_offset = offset;
                return SUCCESS;
            }

            offset += recSize; /* point to next record */
        }
        senv->hdmx_group_offset = 0xFFFFFFFFUL;
        return STATE.error = ERR_xPelHeight_NOT_FOUND;
    }
    else
    {
        FS_BYTE w;
        ttf_read_buf(_PS_ ttf, senv->hdmx_group_offset + 2 + index, 1, (FS_BYTE *)&w);
        *width = (FS_SHORT)w;
        return SUCCESS;
    }
}
#endif /* FS_HINTS */

/****************************************************************/
FS_LONG FSS_get_vdmx_pair(_DS_ FS_SHORT *yMax, FS_SHORT *yMin)
{
    FNTSET *fntset = STATE.cur_typeset.fntset;
    SFNT *sfnt;
    CFNT *cfnt;
    LFNT *lfnt;
    SENV *senv;
    FS_USHORT metric_cmp;

    /* check whether font selected */
    if (fntset == 0)
        return STATE.error = ERR_NO_CURRENT_FNTSET;

    /* get metric font and check whether scale has been set */
    metric_cmp = FNTSET_metric_font_component(fntset);
    cfnt = STATE.cur_typeset.tfntarray[metric_cmp].cfnt;
    sfnt = STATE.cur_typeset.tfntarray[metric_cmp].sfnt;

    if (cfnt == 0 || sfnt == 0)
        return STATE.error = ERR_NO_CURRENT_SFNT;

    lfnt = (LFNT *)(cfnt->lfnt);
    if (lfnt == 0)
    {
        return STATE.error = ERR_NO_CURRENT_LFNT;
    }
    /* restore lfnt, sfnt if needed */
    STATE.cur_lfnt = lfnt;
    STATE.cur_sfnt = sfnt;

    if (check_sfnt(_PS0_)) /* required since senv may have been gutted */
        return STATE.error;

    /* get rotationally invariant xppm, yppm from SENV */
    senv = (SENV *)(sfnt->senv);
    if (!senv)
        return STATE.error = ERR_BAD_SFNT;  /* should not happen */

    /* get vdmx data from the CFNT */
    LFNT_get_vdmx_pair(_PS_ lfnt, senv->xppm, senv->yppm, yMax, yMin);
    if (STATE.error == SUCCESS)
    {
        /* apply component shift adjustment, if any */
        if (sfnt->vertical_shift)
        {
            *yMax += sfnt->vertical_shift;
            *yMin += sfnt->vertical_shift;
        }
    }
    return STATE.error;
}

/****************************************************************/
static FS_LONG
CFNT_get_font_height(_DS_ FNTSET *fntset, CFNT *cfnt, FS_USHORT ppem,
                     FS_USHORT *ascent, FS_USHORT *descent)
{
    FS_FIXED ascender, descender, leading;
    FsAscDescLeadSource source;
    LFNT *lfnt;
    FsPPemAdjust adjustments;

    *ascent = 0;
    *descent = 0;

    lfnt = (LFNT *)(cfnt->lfnt);
    if (lfnt == 0)
    {
        return STATE.error = ERR_NO_CURRENT_LFNT;
    }
    /* Apply linked font scale adjustments */
    CFNT_get_adjustments(_PS_ fntset, cfnt,
                         (FS_FIXED)ppem << 16, &adjustments);
    if (STATE.error)
    {
        return STATE.error;
    }
    if (adjustments.scale)
    {
        if (adjustments.scale < 0)
        {
#ifdef THIS_HAS_A_dotNET_2003_OPTIMIZER_BUG
            ppem -= (FS_USHORT)(-adjustments.scale) >> 8;
#else
            FS_LONG zzzz = (FS_LONG)ppem;
            zzzz += (((FS_LONG)adjustments.scale) / 256);
            ppem = (FS_USHORT)zzzz;
#endif
        }
        else
        {
            ppem += (FS_USHORT)adjustments.scale >> 8;
        }
    }
#if defined(FS_STIK_EVENS_ONLY)
    /* Adjust stik font scale for EVENS ONLY if defined      */
    /* for ESQ Mobile fonts with SmartHint technology        */
    /* and vanilla scale transformations,                    */
    /* if odd size specified, change to next lower even size */
    if (lfnt->fontflags & FONTFLAG_NEW_AA_ON)
    {
        if (ppem & 1) ppem -= 1;
    }
#endif
    LFNT_get_ascender_descender_leading(_PS_ lfnt,
                                        (FS_FIXED)ppem << 16,
                                        &ascender, &descender,
                                        &leading, &source);
    if (STATE.error == SUCCESS)
    {
        FS_FIXED fshift = adjustments.shift << 16;
        if (fshift)
        {
            ascender += fshift;
            descender -= fshift;
        }
        *ascent = FS_ROUND(ascender);
        *descent = FS_ROUND(descender);
        return STATE.error;
    }
    else
    {
        *ascent = 0;
        *descent = 0;
        return STATE.error = ERR_TABLE_NOT_FOUND;
    }
}

/****************************************************************/
FS_USHORT FSS_get_ppem_size(_DS_ FS_USHORT pixel_height, FS_USHORT max_ppem)
{
    FNTSET *fntset = STATE.cur_typeset.fntset;
    TFNT *tfntarray;
    FS_USHORT ppem;
    STATE.error = SUCCESS;

    /* check whether font selected */
    if (!fntset)
    {
        STATE.error = ERR_NO_CURRENT_FNTSET;
        return max_ppem;
    }
    tfntarray = STATE.cur_typeset.tfntarray;
    if (!tfntarray)
    {
        STATE.error = ERR_BAD_TYPESET;
        return 0;
    }

    /* initialize search */
    ppem = max_ppem;

    /* find ppem that height restricts entire linked font */
    while ( ppem > 1 )
    {
        TFNT *tfnt = tfntarray;
        FS_USHORT yMin = 0, yMax = 0;
        FS_USHORT max_height;
        FS_USHORT ascent, descent;
        FS_USHORT num_fonts = fntset->num_fonts;

        while (num_fonts--)
        {
            CFNT_get_font_height(_PS_ fntset, tfnt->cfnt,
                                 ppem, &ascent, &descent);
            if (STATE.error)
                return 0;
            yMax = MAX(yMax, ascent);
            yMin = MAX(yMin, descent);
            tfnt++;
        }
        max_height = yMin + yMax;
        if (max_height > pixel_height)
            ppem--; /* lower ppem and keep searching */
        else
            break;
    }
    return ppem;
}

static FS_VOID adjust_advance(_DS_ SENV *senv, FS_SHORT space,
                              FS_FIXED *dx, FS_FIXED *dy)
{
    FS_FIXED adj_x = 0;
    FS_FIXED adj_y = 0;

    if ( STATE.cur_lfnt->fontflags & FONTFLAG_STIK )
    {
#ifdef FS_PSEUDO_BOLD
        /* adjust escapement for pseudobold */
        if ( (STATE.flags & FLAGS_ADD_WEIGHT) ||
             ( (STATE.cur_lfnt->fontflags & FONTFLAG_NEW_AA_ON) &&
               (STATE.lpm <= 26) && !(STATE.flags & FLAGS_OUTLINEBOLD) ) )
            adj_x = 0; /* no adjustment for CJK bold */
        else
        {
            if (senv->bold_width && !space)
            {
                FIXED_VECTOR bold_vec;
                bold_vec.x = STATE.cur_sfnt->user_scale[0];
                bold_vec.y = STATE.cur_sfnt->user_scale[2];
                fixed_norm(&bold_vec);
                bold_vec.x *= senv->bold_width;
                bold_vec.y *= senv->bold_width;
                adj_x += bold_vec.x;
                adj_y += bold_vec.y;
            }
        }
#else
        space = space;
        senv = senv;
#endif

        /* adjust the escapement */
        *dx += adj_x;
        *dy += adj_y;
    }
}

#ifdef FS_RENDER
/*lint -e438  Warning -- last value assigned to 'id' and 'type' not used    */
static FS_VOID get_outline_advance(_DS_ FS_ULONG index, FS_ULONG id, FS_USHORT type,
                                   FS_SHORT *i_dx, FS_SHORT *i_dy,
                                   FS_FIXED *dx, FS_FIXED *dy)
{
    /* get advance from outline */
    FS_OUTLINE *outl = NULL;
    SENV *senv = (SENV *)(STATE.cur_sfnt->senv);
    FS_ULONG flags = STATE.flags;

#ifdef FS_HINTS
    if ((STATE.cur_lfnt->fontflags & FONTFLAG_STIK) != FONTFLAG_STIK &&
        (STATE.cur_lfnt->fnt_type != PFR_TYPE) &&
        (STATE.cur_sfnt->ref_lines[RTGA_BEEN_THERE] == 0))
        get_ref_lines(_PS0_); /* get RTGAH ref lines since not already done */
#endif

    if (type & FS_MAP_BITMAP)
    {
        STATE.flags &= ~FLAGS_GRAYSCALE;
    }
    else
    {
        STATE.flags |= FLAGS_GRAYSCALE;
    }

#ifdef FS_CACHE_OUTLINES
    if ( !(STATE.flags & FLAGS_RTGAH_REF) )
        outl = find_outline_in_cache(_PS_ index);

    if (outl) /* found in outline cache */
    {
        *dx = outl->dx;
        *dy = outl->dy;
        STATE.any_hints = (outl->outl_flag & OUTL_FLAGS_ANYHINTS) |
                          (outl->outl_flag & OUTL_FLAGS_RTGAH);
        if (outl->outl_flag & OUTL_FLAGS_SCANCTRL)
        {
            STATE.flags |= FLAGS_FIX_DROPOUTS;
            flags  |= FLAGS_FIX_DROPOUTS;
        }
        else
        {
            STATE.flags &= ~FLAGS_FIX_DROPOUTS;
            flags  &= ~FLAGS_FIX_DROPOUTS;
        }

        /* cannot return yet since outline advance may be adjusted */
    }
    else /* if still not found, make outline to get advance */
#endif
    {
        /* make outline to get advance */
        outl = make_outline(_PS_ STATE.cur_sfnt, index);
        if (outl)
        {
            outl->outl_flag = 0;
            outl->outl_flag |= STATE.any_hints;
            if (STATE.flags & FLAGS_FIX_DROPOUTS)
                outl->outl_flag |= OUTL_FLAGS_SCANCTRL;

#ifdef FS_CACHE_OUTLINES
            save_outline_to_cache(_PS_ index, outl);
#endif
            *dx = outl->dx;
            *dy = outl->dy;
            /* do not free outl yet, needed below */
        }
        else
        {
            STATE.flags = flags;
            return;
        }
    }

    STATE.flags = flags;

#ifdef FS_EDGE_RENDER
    STATE.adfGridFitType = ADF_GRID_FIT_NONE;
    if ((STATE.flags & FLAGS_MAZ_ON)) /* user override */
        STATE.adfGridFitType = ADF_GRID_FIT_MAZ_PIXEL;
    else
    {
        if (outl->outl_flag & OUTL_FLAGS_GRIDFITTYPE_MAZ)
            STATE.adfGridFitType = ADF_GRID_FIT_MAZ_PIXEL;
        else if (outl->outl_flag & OUTL_FLAGS_GRIDFITTYPE_BAZ)
            STATE.adfGridFitType = ADF_GRID_FIT_BAZ_PIXEL;
    }
#else
    id = id;
    type = type;
#endif

    /* now possibly adjust escapement depending on type */
    adjust_advance(_PS_ senv, (FS_SHORT)(!outl->num), dx, dy);

#ifdef FS_EDGE_RENDER
    /* adjust escapement for rotated Edge glyphs */
    if ((type & FS_MAP_ANY_EDGE_GRAYMAP) && (*dx && *dy))
    {
        FS_FIXED adj_x = 0;
        FS_FIXED adj_y = 0;
        get_ADF_adjustments(_PS_ outl, id, &adj_x, &adj_y);
        *dx += adj_x;
        *dy += adj_y;
    }
#endif

    /* set integer advance if not rotated */
    if (outl->i_dx)
        *i_dx = FS_ROUND(*dx);
    if (outl->i_dy)
        *i_dy = FS_ROUND(*dy);

    /* free outline since no longer needed */
    if (outl)
        FSS_free_char(_PS_ outl);
}
/*lint +e438  Warning -- last value assigned to 'id' and 'type' not used    */
#endif

#ifdef FS_CACHE_ADVANCE
static FS_VOID cache_advance(_DS_ FS_ULONG index, FS_USHORT type,
                             FS_SHORT i_dx, FS_SHORT i_dy,
                             FS_FIXED dx, FS_FIXED dy)
{
    FS_ADVANCE *advance;

#ifdef FS_MEM_DBG
    STATE.memdbgid = "FS_ADVANCE";
#endif
    advance = FSS_malloc(_PS_ sizeof(FS_ADVANCE));

    if (advance)
    {
        advance->dx = dx;
        advance->dy = dy;
        advance->i_dx = i_dx;
        advance->i_dy = i_dy;
        advance->size = sizeof(FS_ADVANCE);
        advance->cache_ptr = 0;

        save_advance_to_cache(_PS_ index, type, advance);
        FSS_free_char(_PS_ advance);
    }
}
#endif

/****************************************************************/
FS_LONG FSS_get_advance(_DS_ FS_ULONG id, FS_USHORT type,
                        FS_SHORT *i_dx, FS_SHORT *i_dy,
                        FS_FIXED *dx, FS_FIXED *dy)
{
    FS_USHORT index;
    SENV *senv;

    STATE.error = SUCCESS;
    *dx = *dy = 0;
    *i_dx = *i_dy = 0;

    /* character id to index */
    index = map_char(_PS_ id, 0);
    if (STATE.error)
        return STATE.error;

    if (check_sfnt(_PS0_))
        return STATE.error;

    senv = (SENV *)(STATE.cur_sfnt->senv);
    if (senv == 0)return STATE.error = ERR_BAD_SFNT; /* should never happen */

#ifdef FS_ICONS  /* must check icons first if built in (expensive) */
    {
        FS_GLYPHMAP *map = 0;
        TTF *ttf = (TTF *)(STATE.cur_lfnt->fnt);

        /* if index is an icon index, look for icons */
        if (index >= ttf->maxp->numGlyphs)
        {
            if (type & FS_MAP_VECTOR_ICON)
            {
                map = get_icon(_PS_ index, FS_MAP_VECTOR_ICON);
                if (map)
                {
                    *dx = map->dx;
                    *dy = map->dy;
                    *i_dx = map->i_dx;
                    *i_dy = map->i_dy;
                    FSS_free_char(_PS_ map);
                    return SUCCESS;
                }
            }
            if (type & FS_MAP_RASTER_ICON)
            {
                map = get_icon(_PS_ index, FS_MAP_RASTER_ICON);
                if (map)
                {
                    *dx = map->dx;
                    *dy = map->dy;
                    *i_dx = map->i_dx;
                    *i_dy = map->i_dy;
                    FSS_free_char(_PS_ map);
                    return SUCCESS;
                }
            }
        }
    }
#endif /* FS_ICONS */

    /* faster to first check if possibly an hdmx table or unhinted advance */
    if (senv->vanilla && (STATE.cur_lfnt->fnt_type != PFR_TYPE))
    {
        /* look in hdmx table or scale unhinted advance */
        TTF  *ttf = (TTF *)(STATE.cur_lfnt->fnt);
        if (ttf == 0) return STATE.error = ERR_BAD_LFNT; /* should never happen */

        /* check bit 4 in the head table flags to see if linear scaling applies */
        if ( (ttf->head) && !(ttf->head->flags & 0x10) )
        {
#ifdef FS_RENDER
            get_glyph_advance(_PS_ (fsg_SplineKey *)(senv->ttkey), (FS_USHORT)index, dx, dy);
            if (STATE.error) return STATE.error;

#ifdef FS_PSEUDO_BOLD
            if ( !(STATE.cur_lfnt->fontflags & FONTFLAG_STIK) )
                *dx += senv->bold_width << 16;
#endif
#endif
        }

    } /* senv->vanilla */

    if ((*dx != 0) || (*dy != 0))
    {
        /* possibly adjust escapement depending on type */
        adjust_advance(_PS_ senv, 0, dx, dy);

        /* set integer advance if not rotated */
        if (*dx == 0 || *dy == 0)
        {
            *i_dx = FS_ROUND(*dx);
            *i_dy = FS_ROUND(*dy);
        }
        return SUCCESS;  /* no need to cache advance */
    }

    /* try getting advance from cache if available */

#ifdef FS_CACHE_ADVANCE
    {
        FS_ADVANCE *advance;
        advance = find_advance_in_cache(_PS_ index, type);
        if (advance)
        {
            *dx = advance->dx;
            *dy = advance->dy;
            *i_dx = advance->i_dx;
            *i_dy = advance->i_dy;
            FSS_free_char(_PS_ advance);
            return STATE.error;
        }
    }
#else
    type = type;
#endif

    /* try getting advance from various other caches */

#ifdef FS_EDGE_RENDER
#ifdef FS_CACHE_EDGE_GLYPHS
    if (type & FS_MAP_ANY_EDGE_GRAYMAP)
    {
        FS_GRAYMAP *gmap;
        FS_SHORT cache_type = 0;

        if (type & FS_MAP_EDGE_GRAYMAP4)
            cache_type = CACHE_ADF_GRAYMAP;
        else if (type & FS_MAP_EDGE_GRAYMAP8)
            cache_type = CACHE_ADF_GRAYMAP8;
        else if (type & FS_MAP_EDGE_GRAYMAP2)
            cache_type = CACHE_ADF_GRAYMAP2;

        gmap = find_graymap_in_cache(_PS_ index, cache_type, 0, 0);
        if (gmap)
        {
            *dx = gmap->dx;
            *dy = gmap->dy;
            *i_dx = gmap->i_dx;
            *i_dy = gmap->i_dy;
            FSS_free_char(_PS_ gmap);
            return STATE.error;
        }
    }
#endif
#endif

#ifdef FS_CACHE_GRAYMAPS
    if (type & FS_MAP_ANY_GRAYMAP)
    {
        FS_GRAYMAP *gmap;
        FS_SHORT cache_type = 0;
        if (type & FS_MAP_GRAYMAP4)
            cache_type = CACHE_GRAYMAP;
        else if (type & FS_MAP_GRAYMAP8)
            cache_type = CACHE_GRAYMAP8;
        else if (type & FS_MAP_GRAYMAP2)
            cache_type = CACHE_GRAYMAP2;

        gmap = find_graymap_in_cache(_PS_ index, cache_type, 0, 0);
        if (gmap)
        {
            *dx = gmap->dx;
            *dy = gmap->dy;
            *i_dx = gmap->i_dx;
            *i_dy = gmap->i_dy;
            FSS_free_char(_PS_ gmap);
            return STATE.error;
        }
    }
#endif

#ifdef FS_CACHE_BITMAPS
    if (type & FS_MAP_BITMAP)
    {
        FS_BITMAP *bmap;
        bmap = find_bitmap_in_cache(_PS_ index);
        if (bmap)
        {
            *dx = bmap->dx;
            *dy = bmap->dy;
            *i_dx = bmap->i_dx;
            *i_dy = bmap->i_dy;
            FSS_free_char(_PS_ bmap);
            return STATE.error;
        }
    }
#endif

#ifdef FS_EMBEDDED_BITMAP

    /* check for embedded graymap */
    if ((type & FS_MAP_ANY_GRAYMAP) &&
        !(STATE.flags & (FLAGS_EMBOSSED | FLAGS_ENGRAVED | FLAGS_OUTLINED | FLAGS_OUTLINED_2PIXEL |
                         FLAGS_OUTLINED_UNFILLED | FLAGS_OUTLINED_FILLED)))
    {
        FS_GRAYMAP *gmap;
        gmap = get_embedded_graymap(_PS_ STATE.cur_sfnt, (FS_USHORT)index, type);
        if (gmap)
        {
            *dx = gmap->dx;
            *dy = gmap->dy;
            *i_dx = gmap->i_dx;
            *i_dy = gmap->i_dy;
            FSS_free_char(_PS_ gmap);
            return STATE.error;
        }
    }

    /* check for embedded bitmap */
    if ( (type & FS_MAP_BITMAP) && 
         !(type & FS_MAP_ANY_EDGE_GRAYMAP) &&
         !(type & FS_MAP_ANY_GRAYMAP) )
    {
        FS_BITMAP *bmap;
        bmap = get_embedded_bitmap(_PS_ STATE.cur_sfnt, (FS_USHORT)index);
        if (bmap)
        {
            *dx = bmap->dx;
            *dy = bmap->dy;
            *i_dx = bmap->i_dx;
            *i_dy = bmap->i_dy;
            FSS_free_char(_PS_ bmap);
            return STATE.error;
        }
    }

#endif

#ifdef FS_HINTS
    /* if hdmx table, try getting advance from it */
    if ( !(STATE.flags & FLAGS_HINTS_OFF))
    {
        FS_SHORT width;

        /* get hdmx data from the LFNT */
        if (senv->hdmx_group_offset != 0xFFFFFFFFUL)
        {
            SFNT_get_hdmx_width(_PS_ STATE.cur_sfnt, index, &width);
            if (STATE.error == SUCCESS)
            {
                *dx = (FS_FIXED)(width << 16);
#ifdef FS_PSEUDO_BOLD
                *dx += (FS_FIXED)(senv->bold_width) << 16;
#endif
            } /* success */
        }
        STATE.error = SUCCESS; /* ok if not found in hdmx table */
    }
#endif

    if ((*dx == 0) || (*dy == 0))
    {
        /* get advance from outline if rendering enabled */
#ifdef FS_RENDER
        get_outline_advance(_PS_ index, id, type, i_dx, i_dy, dx, dy);
#endif
    }
    else
    {
        /* possibly adjust escapement depending on type */
        adjust_advance(_PS_ senv, 0, dx, dy);

        /* set integer advance if not rotated */
        if(*dx == 0 || *dy == 0)
        {
            *i_dx = FS_ROUND(*dx);
            *i_dy = FS_ROUND(*dy);
        }
    }

    /* save advance to cache */
#ifdef FS_CACHE_ADVANCE
    cache_advance(_PS_ index, type, *i_dx, *i_dy, *dx, *dy);
#endif

    return STATE.error;
}

/****************************************************************/
FS_LONG FSS_resize(_DS_ FS_ULONG nsize)
{
#ifdef FS_INT_MEM
    if (nsize == STATE.server->heap_size)
    {
        return STATE.error = SUCCESS;
    }
    else
    {
        /* just can't resize internal memory */
        return STATE.error = ERR_RESIZE_FAIL;
    }
#else
    if (nsize >= STATE.server->heap_size)
    {
        /* upsizing is trivial */
        STATE.server->heap_size = nsize;
        return STATE.error = SUCCESS;
    }
    else
    {
        /* downsizing is a little worse */
        int ok;

        while (STATE.server->allocated > nsize)
        {
            ok = get_some_back(_PS0_);
            if (!ok)
            {
                return STATE.error = ERR_RESIZE_FAIL;
            }
        }
        STATE.server->heap_size = nsize;
        return STATE.error = SUCCESS;
    }
#endif    /* FS_INT_MEM */
}

/****************************************************************/
FS_ULONG FS_get_version(_DS0_)
{
    return STATE.server->version;
}

/****************************************************************/
FS_ULONG FS_get_heap_used(_DS0_)
{
    if ( STATE.server == 0 )
        return 0;

    return STATE.server->allocated;
}

/****************************************************************/
/* translate character id to glyph index via current CMAP */
FS_USHORT FSS_map_char(_DS_ FS_ULONG id)
{
    FS_USHORT index;
    FS_ULONG flags = STATE.flags;

    STATE.flags &= FLAGS_CMAP_ON;
    index = map_char(_PS_ id, 0);
    if (STATE.error == 0)
        index += STATE.cur_typeset.tfntarray[STATE.cur_font].cfnt->start_index;
    STATE.flags = flags;
    return index;
}

FS_LONG FSS_get_gpos_scale(_DS_ FS_USHORT id, FS_USHORT *du,
                           FS_FIXED *s00, FS_FIXED *s01, FS_FIXED *s10, FS_FIXED *s11,
                           FS_FIXED *xppm, FS_FIXED *yppm,
                           FS_USHORT *idRangeStart, FS_USHORT *idRangeEnd)
{
    FS_LONG ret;
    FS_FIXED tan_s;
    FS_ULONG flags = STATE.flags;
    STATE.flags |= FLAGS_CMAP_OFF;
    map_char(_PS_ id, 0);
    STATE.flags = flags;

    if (STATE.error) return STATE.error;

    if (check_sfnt(_PS0_)) return STATE.error;

    *du = FSS_get_design_units(_PS0_);

    if (STATE.error) return STATE.error;

    *s00 = STATE.cur_sfnt->user_scale[0];
    *s01 = STATE.cur_sfnt->user_scale[1];
    *s10 = STATE.cur_sfnt->user_scale[2];
    *s11 = STATE.cur_sfnt->user_scale[3];

    ret = get_scale_inputs(STATE.cur_sfnt->user_scale, xppm, yppm, &tan_s);

    *idRangeStart = STATE.cur_typeset.tfntarray[STATE.cur_font].cfnt->start_index;
    if ((STATE.cur_font + 1) < STATE.cur_typeset.fntset->num_fonts)
    {
        *idRangeEnd = STATE.cur_typeset.tfntarray[STATE.cur_font + 1].cfnt->start_index - 1;
    }
    else
    {
        *idRangeEnd = 0xFFFF;
    }
    return ret;
}

/****************************************************************/
/* translate glyph index to character id via current CMAP */
FS_ULONG FSS_inverse_map_char(_DS_ FS_ULONG gi)
{
    FS_ULONG id;
    id = inverse_map_char(_PS_ gi, NULL);
    return id;
}
/****************************************************************/
/* get variant glyph index for character id via format 14 cmap  */
FS_USHORT FSS_map_char_variant(_DS_ FS_ULONG id, FS_ULONG varSelector)
{
    FS_USHORT index;
    FS_ULONG flags = STATE.flags;

    STATE.flags &= FLAGS_CMAP_ON;
    if (varSelector)
    {
        FS_USHORT platform = STATE.platform;
        FS_USHORT encoding = STATE.encoding;

        FSS_set_cmap(_PS_ STATE.varPlatform, STATE.varEncoding);
        if (STATE.error)
        {
            FSS_set_cmap(_PS_ platform, encoding);
            STATE.flags = flags;
            return 0; /* variant cmap not found */
        }
        else
        {
            index = map_char(_PS_ id, varSelector);
            if (STATE.error == 0)
                index += STATE.cur_typeset.tfntarray[STATE.cur_font].cfnt->start_index;
            FSS_set_cmap(_PS_ platform, encoding);
            STATE.flags = flags;

            return index; /* variation index or zero if not found */
        }
    }

    /* return default variation */
    index = map_char(_PS_ id, 0);
    if (STATE.error == 0)
        index += STATE.cur_typeset.tfntarray[STATE.cur_font].cfnt->start_index;
    STATE.flags = flags;
    return index;
}

/****************************************************************/

FS_LONG load_fnt(_DS_ LFNT *lfnt)
{
    FILECHAR *path;
    FS_BYTE *memptr;

    /* clear some font flags */
    lfnt->fontflags &= ~ FONTFLAG_STIK;
    lfnt->fontflags &= ~ FONTFLAG_ACT3;
    lfnt->fontflags &= ~ FONTFLAG_CCC;
    lfnt->fontflags &= ~ FONTFLAG_DDD;
    lfnt->fontflags &= ~ FONTFLAG_EDGE;

    if (lfnt->path && !lfnt->memptr)
    {
#ifdef FS_MAPPED_FONTS
        path = NULL;
        memptr = MF_get_mapped_font(_PS_ lfnt->path);
        if (!memptr)
        {
            return STATE.error;
        }
        lfnt->memptr = memptr;

#else
        path = lfnt->path;
        memptr = NULL;
#endif
    }
    else
    {
        path = NULL;
        memptr = lfnt->memptr;
    }
    /* set loading flag so that get_some_back */
    /* will not release the ttf for current lfnt */
    lfnt->loading = 1;

    if ((lfnt->fnt_type == TTF_TYPE) || 
        (lfnt->fnt_type == TTC_TYPE) ||
        (lfnt->fnt_type == CFF_TYPE)  )
    {
#ifndef FS_CFFR
        if (lfnt->fnt_type == CFF_TYPE)
        {
            lfnt->fnt = NULL;
            STATE.error = ERR_CFF_UNDEF;
            /* restore the loading flag to its original value */
            lfnt->loading = 0;
            return STATE.error;
        }
#endif
        lfnt->fnt = load_ttf(_PS_
                             path,
                             memptr,
                             lfnt->index,
                             lfnt->data_offset,
                             lfnt->data_length, 0 /* no validate */);

        load_cmap_cache(_PS_ lfnt);

        /* set some fontflags */
        if(lfnt->fnt)
        {
            FS_SHORT format;
            TTF *ttf = (TTF *)lfnt->fnt;

#ifdef FS_DEFERRED_LOAD_DBG
            FS_PRINTF(("\n%% load font %s\n", (FILECHAR*) ttf->name->font_name));
#endif
            /* set some flags */
            format = ttf->head->glyphDataFormat;

#ifdef FS_EDGE_TECH
            if(ttf->adfh != 0)
                lfnt->fontflags = FONTFLAG_EDGE;  /* only applies if Edge is enabled */
#endif
            if (ttf->hhea->horizontalCaretSlopeDenominator == 0)
                lfnt->fontflags &= ~ FONTFLAG_IS_ITALIC;
            else
                lfnt->fontflags |= FONTFLAG_IS_ITALIC;

            if (STIK_FORMAT_CCC(format))
            {
                lfnt->fontflags |= FONTFLAG_STIK;
                lfnt->fontflags |= FONTFLAG_CCC;
            }
            else if (STIK_FORMAT_DDD(format))
            {
                lfnt->fontflags |= FONTFLAG_STIK;
                lfnt->fontflags |= FONTFLAG_DDD;
#ifdef FS_EDGE_TECH
                if(ttf->adfh == 0)
                    STATE.error = ERR_NOT_SUPPORTED;
#else
                STATE.error = ERR_NOT_SUPPORTED;
#endif
            }
            else if ((format == STIK_FORMAT) ||
                     (format == OFFLINE_STIK_FORMAT))
            {
                lfnt->fontflags |= FONTFLAG_STIK;
            }
            else if (TTF_FORMAT_CCC(format))
            {
                lfnt->fontflags |= FONTFLAG_CCC;
            }
            else if (TTF_FORMAT_DDD(format))
            {
                lfnt->fontflags |= FONTFLAG_DDD;
#ifdef FS_EDGE_TECH
                if(ttf->adfh == 0)
                    STATE.error = ERR_NOT_SUPPORTED;
#else
                STATE.error = ERR_NOT_SUPPORTED;
#endif
            }
            if ((format == OFFLINE_STIK_FORMAT) ||
                (format == OFFLINE_CCC_FORMAT))
            {
                lfnt->fontflags |= FONTFLAG_NEW_AA_ON;
                lfnt->fontflags |= FONTFLAG_DIRECT_DRAW_ON;
            }
            if (((TTF *)lfnt->fnt)->decomp)
            {
                lfnt->fontflags |= FONTFLAG_ACT3;
            }
        }

#ifdef FS_ICONS
        if (lfnt->fnt)
        {
            TTF *ttf = (TTF *)(lfnt->fnt);
            if (ttf->icon_offset)
            {
                /* see how many icons there are */
                ttf_read_buf(_PS_ ttf, ttf->icon_offset + 8, 4, (FS_BYTE *)&ttf->num_icons);
                ttf->num_icons = SWAPL(ttf->num_icons);

                if (ttf->num_icons)
                {
                    FS_ULONG *entries;
                    /* read entries array */
#ifdef FS_MEM_DBG
                    STATE.memdbgid = "icon->entries";
#endif
                    entries = (FS_ULONG *)ttf_read(_PS_ ttf, ttf->icon_offset + 12, 4 * ttf->num_icons);
                    if (entries && !STATE.error)
                    {
                        /* store first and last index for fast index look up */
                        ttf->icon_first = SWAPL(entries[0]) >> 16;
                        ttf->icon_last = SWAPL(entries[ttf->num_icons - 1]) >> 16;
                    }
                    FSS_free(_PS_ entries);
                }
                else
                {
                    ttf->icon_first = 0;
                    ttf->icon_last  = 0;
                }
            }
            else
            {
                ttf->num_icons  = 0;
                ttf->icon_first = 0;
                ttf->icon_last  = 0;
            }
        }
#endif /* FS_ICONS */
    }

    else if (lfnt->fnt_type == PFR_TYPE)
    {
#ifdef FS_PFRR

        /* file based font ? */
        if (path)
        {
            /* try as vanilla file */
            FS_FILE *fp = FS_open(_PS_ path); /* closed in unload_pfr */
            if (fp == 0)
                return 0;
            lfnt->fnt = load_pfr(_PS_ path, memptr, lfnt->index, fp);
        }
        else if (memptr)
        {
            lfnt->fnt = load_pfr(_PS_ NULL, memptr, lfnt->index, NULL);
        }
        else
        {
            lfnt->fnt = NULL;
            STATE.error = ERR_BAD_LFNT;
        }
#else
        lfnt->fnt = NULL;
        STATE.error = ERR_PFR_UNDEF;
#endif
    }
    else
    {
        lfnt->fnt = NULL;
        STATE.error = ERR_BAD_LFNT;
    }

    /* restore the loading flag to its original value */
    lfnt->loading = 0;

    return STATE.error;
}

/****************************************************************/
FS_BYTE fontfile_type(_DS_ FILECHAR *path,
                      FS_BYTE *memptr,
                      FS_ULONG data_offset)
{
    FILECHAR buf[4];

#ifdef FS_MAPPED_FONTS
    if (path && !memptr)
    {
        memptr = MF_get_mapped_font(_PS_ path);
        if (memptr)
        {
            path = NULL;
        }
    }
#endif
    /* memory based font ? */
    if (memptr)
    {
        SYS_MEMCPY(buf, memptr + data_offset, 3);
    }
    /* file based font ? */
    else if (path)
    {
        FS_FILE *fp;

        /* try as file */
        fp = FS_open(_PS_ path);
        if (!fp)
        {
            return UNK_TYPE;
        }
        else
        {
            if (FS_seek(_PS_ fp, data_offset, SEEK_SET))
            {
                FS_close(_PS_ fp);
                return UNK_TYPE;
            }
            if (FS_read(_PS_ fp, (FS_BYTE *)buf, 3) != 3)
            {
                FS_close(_PS_ fp);
                return UNK_TYPE;
            }
            FS_close(_PS_ fp);
        }
    }
    buf[3] = '\0';

    if (buf[0] == 'P')
    {
        return PFR_TYPE;
    }
    if (buf[0] == 'O' && buf[1] == 'T' && buf[2] == 'T')
    {
        return CFF_TYPE;
    }
    else
    {
        if (FS_streq(buf, "LTT"))
        {
            return LTT_TYPE;
        }
        else
        {
            return TTF_TYPE;
        }
    }
}

/****************************************************************/
FS_VOID unload_cmap_cache(_DS_ LFNT *lfnt)
{
#ifdef FS_CACHE_CMAP
    if (lfnt->cmap_cache != 0)
    {
        FSS_free(_PS_ (FS_CMAP_CACHE *)(lfnt->cmap_cache));
        lfnt->cmap_cache = 0;
    }
#else
    lfnt = lfnt;
    FS_state_ptr = FS_state_ptr;
#endif
    return;
}

/****************************************************************/
FS_VOID unload_fnt(_DS_ LFNT *lfnt)
{
    if (lfnt->fnt)
    {
        if ((lfnt->fnt_type == TTF_TYPE) || 
            (lfnt->fnt_type == TTC_TYPE) ||
            (lfnt->fnt_type == CFF_TYPE) )
        {
            unload_ttf(_PS_ (TTF *)(lfnt->fnt));
            unload_cmap_cache(_PS_ lfnt);
        }
#ifdef FS_PFRR
        else if (lfnt->fnt_type == PFR_TYPE)
        {
            unload_pfr(_PS_ (PFR *)(lfnt->fnt));
        }
#endif
        lfnt->fnt = 0;
    }
}

/****************************************************************/
FS_LONG scale_font(_DS_ SFNT *sfnt, FS_FIXED s00, FS_FIXED s01, FS_FIXED s10, FS_FIXED s11)
{
    LFNT *lfnt = (LFNT *)(sfnt->lfnt);

    if (!lfnt)
        return STATE.error = ERR_NO_CURRENT_LFNT;

    if ((lfnt->fnt_type == TTF_TYPE) ||
            (lfnt->fnt_type == TTC_TYPE))
    {
        return scale_font_ttf(_PS_ sfnt, s00, s01, s10, s11);
    }
    else if (lfnt->fnt_type == CFF_TYPE)
    {
        return scale_font_ttf(_PS_ sfnt, s00, s01, s10, s11);
    }
    else
    {
#ifdef FS_PFRR
        return scale_font_pfr(_PS_ sfnt, s00, s01, s10, s11);
#else
        return STATE.error = ERR_PFR_UNDEF;
#endif
    }
}

#ifdef FS_RENDER
/****************************************************************/
FS_OUTLINE *make_outline(_DS_ SFNT *sfnt, FS_ULONG gIndex)
{
    LFNT *lfnt = (LFNT *)(sfnt->lfnt);

    if ((lfnt->fnt_type == TTF_TYPE) ||
        (lfnt->fnt_type == TTC_TYPE) ||
        (lfnt->fnt_type == CFF_TYPE))
        return make_outline_ttf(_PS_ sfnt, gIndex);
    else
    {
#ifdef FS_PFRR
        return make_outline_pfr(_PS_ sfnt, gIndex);
#else
        STATE.error = ERR_PFR_UNDEF;
        return NULL;
#endif
    }
}
#endif /* FS_RENDER */

/****************************************************************/
FS_STATE *FSS_new_client(FS_STATE *sp, FS_ULONG heap_size)
{
    if (heap_size > 0)
    {
        FSS_resize(sp, sp->server->heap_size + heap_size);
        if (sp->error)
        {
            return 0;
        }
    }
    {
        FS_STATE *state;

        state = (FS_STATE *)FSS_malloc(sp, sizeof(FS_STATE));
        if (!state)
        {
            sp->error = ERR_MALLOC_FAIL;
            return 0;
        }
        FS_STATE_init(state);

        state->parent = sp;

        state->parent->ref_count++;

        state->heap_size = heap_size;

        state->server = sp->server;

        state->server->client_count++;

#ifdef FS_MEM_DBG
        state->memdbgfp = sp->memdbgfp;
#endif

#ifdef FS_EDGE_TECH
        startADF(state);
#endif
        return state;
    }
}

/****************************************************************/
FS_ULONG FSS_end_client(FS_STATE *sp)
{
    if (sp == NULL)
        return ERR_INVALID_ADDRESS;

    if (sp->ref_count)
    {
        return sp->error = ERR_BAD_REF_COUNT;
    }
    else
    {
        FS_STATE_done(sp);
        if (sp->error)
        {
            return sp->error;
        }
        else
        {
            if (sp->heap_size > 0)
            {
                FSS_resize(sp, sp->server->heap_size - sp->heap_size);
                if (sp->error)
                {
                    return sp->error;
                }
            }
            sp->parent->ref_count--;

            sp->server->client_count--;

            FSS_free(sp->parent, sp);

            return SUCCESS;
        }
    }
}

static FS_STYLE
FNTSET_getStyle(_DS_ FNTSET *fntset)
{
    LFNT *lfnt;
    FS_VOID *fnt;

    lfnt = (LFNT *)(fntset->metric_font);

    /* lfnt->fnt must exist as well */
    if (!lfnt->fnt && (load_fnt(_PS_ lfnt) != SUCCESS))
    {
        return STYLE_ANY;
    }
    fnt = (FS_VOID *)(lfnt->fnt);

    if ((lfnt->fnt_type == TTF_TYPE) ||
            (lfnt->fnt_type == TTC_TYPE))
    {
        TTF *ttf;
        TTF_HEAD *phead;

        ttf = fnt;

        phead = (TTF_HEAD *)(ttf->head);

        /* From TrueType specification:
            Bit 0: Bold (if set to 1);
            Bit 1: Italic (if set to 1)
            Bit 2: Underline (if set to 1)
            Bit 3: Outline (if set to 1)
            Bit 4: Shadow (if set to 1)
            Bit 5: Condensed (if set to 1)
            Bit 6: Extended (if set to 1)
            Bits 7-15: Reserved (set to 0).
         */
        /* From TrueType specification:
            Note that the macStyle bits must agree with the 'OS/2' table
            fsSelection bits. The fsSelection bits are used over the macStyle
            bits in Microsoft Windows. The PANOSE values and 'post' table
            values are ignored for determining bold or italic fonts.
         */
        switch (phead->macStyle & 0x3)
        {
        default:
        case 0x0:
            return STYLE_PLAIN;
        case 0x1:
            return STYLE_BOLD;
        case 0x2:
            return STYLE_ITALIC;
        case 0x3:
            return STYLE_BOLDITALIC;
        }
    }
    else
    {
#ifdef FS_PFRR
        return STYLE_PLAIN;
#else
        STATE.error = ERR_PFR_UNDEF;
        return STYLE_ANY;
#endif
    }
}

static FS_FACE
FNTSET_getFace(_DS_ FNTSET *fntset)
{
    LFNT *lfnt;
    FS_VOID *fnt;

    lfnt = (LFNT *)(fntset->metric_font);

    /* lfnt->fnt must exist as well */
    if (!lfnt->fnt && (load_fnt(_PS_ lfnt) != SUCCESS))
    {
        return FACE_ANY;
    }
    fnt = (FS_VOID *)(lfnt->fnt);

    if ((lfnt->fnt_type == TTF_TYPE) ||
            (lfnt->fnt_type == TTC_TYPE))
    {
        TTF *ttf = fnt;

#ifdef MONOSPACED_VIA_HHEA
        TTF_HHEA *phhea;

        /* From TrueType 'hmtx' table doc (concerning 'numOfHMetrics'):
            The value numOfHMetrics comes from the 'hhea' table.
            If the font is monospaced, only one entry need be in the array,
            but that entry is required. The last entry applies to all
            subsequent glyphs.
         */
        phhea = (TTF_HHEA *)(ttf->hhea);

        if (phhea->numberOf_LongHorMetrics > 1)
        {
            return FACE_PROPORTIONAL;
        }
        else
        {
            return FACE_MONOSPACED;
        }
#endif
#ifdef MONOSPACED_VIA_POST
        FS_BYTE *table;
        FS_ULONG length;
        FS_FACE retval;

        /* From TrueType 'post' table doc (concerning 'isFixedPitch'):
            Set to 0 if the font is proportionally spaced, non-zero if the
            font is not proportionally spaced (i.e. monospaced).
         */
        /* this require the fntset be the "cur_fntset" in STATE. Yuck! */
        table = FSS_get_table(_PS_ TAG_post, TBL_EXTRACT, &length);
        if (table)
        {
            TTF_POST post;

            post_init(&post, table, table);

            if (post.isFixedPitch)
            {
                retval = FACE_MONOSPACED;
            }
            else
            {
                retval = FACE_PROPORTIONAL;
            }
            FSS_free_table(_PS_ table);
        }
        else retval = FACE_ANY;

        return retval;
#endif
#ifndef DO_NOT_DO_MONOSPACED_VIA_OS_2
        TTF_OS2 *pos2;

        /* From TrueType 'os2' table doc (concerning 'panose'):
            The PANOSE definition contains ten digits each of which currently
            describes up to sixteen variations. Windows uses bFamilyType,
            bSerifStyle and bProportion in the font mapper to determine
            family type. It also uses bProportion to determine if the font
            is monospaced.
         */
        pos2 = (TTF_OS2 *)(ttf->os2);
        if (pos2)
        {
            if ((pos2->panose[0] == 2) || (pos2->panose[0] == 4))
            {
                if (pos2->panose[3] == 9 /* monospaced */)
                {
                    return FACE_MONOSPACED;
                }
                else
                {
                    return FACE_PROPORTIONAL;
                }
            }
            else if ((pos2->panose[0] == 3) || (pos2->panose[0] == 5))
            {
                if (pos2->panose[3] == 3 /* monospaced */)
                {
                    return FACE_MONOSPACED;
                }
                else
                {
                    return FACE_PROPORTIONAL;
                }
            }
            else return FACE_ANY;
        }
        else
            return FACE_ANY;
#endif
    }
    else
    {
#ifdef FS_PFRR
        return FACE_PROPORTIONAL;
#else
        STATE.error = ERR_PFR_UNDEF;
        return FACE_ANY;
#endif
    }
}

#define BUFGROWTH 256

static FS_LONG
FS_STATE_append_font_names(FS_STATE *sp,
                           FILECHAR **buffer, FS_ULONG *bufLen,
                           FILECHAR **next,
                           FS_STYLE style, FS_FACE face)
{
    ADDEDFNT *fnt;

    for (fnt = sp->name_map; fnt; fnt = fnt->next)
    {
        FS_ULONG len;

        if (style != STYLE_ANY)
        {
            FS_STYLE myStyle = FNTSET_getStyle(sp, fnt->set);
            if (myStyle != style) continue;
        }
        if (face != FACE_ANY)
        {
            FS_FACE myFace = FNTSET_getFace(sp, fnt->set);
            if (myFace != face) continue;
        }
        len = SYS_STRLEN(fnt->name);

        if (*bufLen < ((*next - *buffer) + len + 1 + 1))
        {
            FS_ULONG offset = *next - *buffer;

            FILECHAR *newBuf = FSS_realloc(sp, *buffer, *bufLen + BUFGROWTH);
            if (!newBuf)
            {
                return ERR_MALLOC_FAIL;
            }
            *buffer = newBuf;
            *bufLen += BUFGROWTH;

            *next = *buffer + offset;
        }
        SYS_STRNCPY(*next, fnt->name, len);
        (*next)[len] = '\0';

        *next += len + 1;

        **next = '\0';
    }
    return SUCCESS;
}

static FS_LONG
FS_STATE_all_font_names(_DS_ FILECHAR **buffer, FS_ULONG *bufLen,
                        FS_STYLE style, FS_FACE face)
{
    FS_STATE *sp;
    FILECHAR *next;

    /* This function should not be called if using internal memory and a client-supplied heap */
#ifdef FS_INT_MEM
    if (FS_state_ptr->server->iTypeOwnsHeap == false)
    {
        return ERR_MALLOC_FAIL;
    }
#endif

    *buffer = SYS_MALLOC(BUFGROWTH);
    if (!*buffer)
    {
        *bufLen = 0;
        return ERR_MALLOC_FAIL;
    }
    *bufLen = BUFGROWTH;

    next = *buffer;

    *next = '\0';
    *(next + 1) = '\0';

    sp = &STATE;

    do
    {
        if (FS_STATE_append_font_names(sp, buffer,
                                       bufLen, &next, style, face) != SUCCESS)
        {
            return ERR_MALLOC_FAIL;
        }
        sp = sp->parent;

    }
    while (sp);

    return SUCCESS;
}

static FS_VOID
nil_to_comma(FILECHAR *buffer)
{
    FILECHAR *buffer2 = buffer;
    do
    {
        while (*buffer2) buffer2++;

        *buffer2++ = ',';

    }
    while (*buffer2++);

    if (buffer2 >= (buffer + 2)) buffer2[-2] = '\0';
}

/* The font names will be returned in a byte string containing a
   comma-separated list of font names. The string will be NULL-terminated
   and will not exceed the length returned as an argument.
*/
FS_LONG
FSS_get_available_fonts(_DS_ FS_STYLE style, FS_FACE face,
                        FILECHAR **list, FS_ULONG *length)
{
    /* This function should not be called if using internal memory and a client-supplied heap */
#ifdef FS_INT_MEM
    if (FS_state_ptr->server->iTypeOwnsHeap == false)
    {
        return ERR_MALLOC_FAIL;
    }
#endif

    if (FS_STATE_all_font_names(_PS_ list, length, style, face) != SUCCESS)
    {
        return ERR_MALLOC_FAIL;
    }

    /* need to retain current "STATE" - cur_lfnt, etc. */

    nil_to_comma(*list);

    return SUCCESS;
}

#ifdef FS_RENDER
#ifdef FS_EXTERNAL_OUTLINE

static FS_LONG
check_outline(_DS_ FS_OUTLINE *outl, FS_BOOLEAN doScaling)
{
    if (!doScaling)
    {
        FS_BYTE *t_src;
        FS_FIXED *x, *y;
        int i;
        FS_FIXED min = -4096 << 16;
        FS_FIXED max =  4096 << 16;

        t_src = outl->type;
        x = outl->x;
        y = outl->y;

        for (i = 0; i < outl->num; i++)
        {
            int p, npts;

            switch (*t_src++ & 0x7f)
            {
            case FS_MOVETO:
            case FS_LINETO: /* 1 */
                npts = 1;
                break;
            case FS_QUADTO: /* 2 */
                npts = 2;
                break;
            case FS_CUBETO: /* 3 */
                npts = 3;
                break;
            default:
                npts = 0;
                break;
            }
            for (p = 0; p < npts; p++)
            {
                if ((*x < min) || (*x > max) ||
                        (*y < min) || (*y > max))
                    return STATE.error = POINTS_DATA_ERR;
                x++;
                y++;
            }
        }
    }
    return STATE.error = SUCCESS;
}

static FS_VOID
transform_user_point(FS_FIXED *s,
                     FS_FIXED *x, FS_FIXED *y, FS_USHORT unitsPerEm)
{
    FS_FIXED nx, ny;

    nx = FixMul(s[0], *x) + FixMul(s[1], *y);
    ny = FixMul(s[2], *x) + FixMul(s[3], *y);
    *x = nx / unitsPerEm;
    *y = ny / unitsPerEm;
}

static FS_VOID
scale_user_outline(_DS_ FS_USHORT unitsPerEm)
{
    FS_BYTE *t_src;
    FS_FIXED *x, *y;
    FS_FIXED *s;
    int i;

    t_src = (FS_BYTE *)(STATE.user_outline->type);
    x = (FS_FIXED *)(STATE.user_outline->x);
    y = (FS_FIXED *)(STATE.user_outline->y);

    s = STATE.user_outline_scale;

    for (i = 0; i < STATE.user_outline->num; i++)
    {
        switch (*t_src++ & 0x7f)
        {
        case FS_MOVETO:
        case FS_LINETO: /* 1 */
            transform_user_point(s, x++, y++, unitsPerEm);
            break;
        case FS_QUADTO: /* 2 */
            transform_user_point(s, x++, y++, unitsPerEm);
            transform_user_point(s, x++, y++, unitsPerEm);
            break;
        case FS_CUBETO: /* 3 */
            transform_user_point(s, x++, y++, unitsPerEm);
            transform_user_point(s, x++, y++, unitsPerEm);
            transform_user_point(s, x++, y++, unitsPerEm);
            break;
        default:
            break;
        }
    }
}
/*lint -e569  Warning 569: Loss of information (assignment) (32 bits to 31 bits) */
static FS_VOID
set_user_outline_bounds(_DS0_)
{
    FS_BYTE *t_src;
    FS_FIXED *x, *y;
    FS_FIXED xlo, ylo, xhi, yhi;
    int i;

    t_src = (FS_BYTE *)(STATE.user_outline->type);
    x = (FS_FIXED *)(STATE.user_outline->x);
    y = (FS_FIXED *)(STATE.user_outline->y);

    xlo = ylo = 0x7fffffff;
    xhi = yhi = 0x80000000;

    for (i = 0; i < STATE.user_outline->num; i++)
    {
        int p, npts;

        switch (*t_src++ & 0x7f)
        {
        case FS_MOVETO:
        case FS_LINETO: /* 1 */
            npts = 1;
            break;
        case FS_QUADTO: /* 2 */
            npts = 2;
            break;
        case FS_CUBETO: /* 3 */
            npts = 3;
            break;
        default:
            npts = 0;
            break;
        }
        for (p = 0; p < npts; p++)
        {
            if (*x < xlo) xlo = *x;
            if (*x > xhi) xhi = *x;
            if (*y < ylo) ylo = *y;
            if (*y > yhi) yhi = *y;
            x++;
            y++;
        }
    }
    if (x != (FS_FIXED *)(STATE.user_outline->x))
    {
        STATE.user_outline->lo_x = xlo;
        STATE.user_outline->hi_x = xhi;
        STATE.user_outline->lo_y = ylo;
        STATE.user_outline->hi_y = yhi;
    }
    else
    {
        STATE.user_outline->lo_x = 0;
        STATE.user_outline->hi_x = 0;
        STATE.user_outline->lo_y = 0;
        STATE.user_outline->hi_y = 0;
    }
}
/*lint +e569  Warning 569: Loss of information (assignment) (32 bits to 31 bits) */
#endif /* FS_EXTERNAL_OUTLINE */
#endif /* FS_RENDER */

FS_LONG
FSS_set_outline(_DS_ FS_OUTLINE *outl, FS_BOOLEAN doScaling,
                FS_USHORT unitsPerEm)
{
#ifdef FS_RENDER
#ifdef FS_EXTERNAL_OUTLINE
    FS_BOOLEAN need_to_set_advance = 0;

    if (STATE.user_outline)
    {
        FSS_free_char(_PS_ STATE.user_outline);
        STATE.user_outline = NULL;
    }
    if (!outl) return STATE.error = SUCCESS;

    if (check_outline(_PS_ outl, doScaling) != SUCCESS)
    {
        return STATE.error;
    }
    if (!outl->lo_x && !outl->hi_x &&
        !outl->lo_y && !outl->hi_y &&
        !outl->i_dx && !outl->i_dy &&
        !outl->dx   && !outl->dy) 
        need_to_set_advance = 1;

    STATE.user_outline = copy_outline(_PS_ outl,
                                      1 /* source from user mem */);
    if (!STATE.user_outline)
    {
        return STATE.error;
    }
    if (doScaling)
    {
        FS_FIXED *s;
        FS_BOOLEAN identity;
        SFNT *sfnt;

        if (check_sfnt(_PS0_))
            return STATE.error;

        sfnt = STATE.cur_sfnt;

        s = sfnt->user_scale;

        STATE.user_outline_scale[0] = s[0];
        STATE.user_outline_scale[1] = s[1];
        STATE.user_outline_scale[2] = s[2];
        STATE.user_outline_scale[3] = s[3];

        identity = ((STATE.user_outline_scale[0] == FIXED_ONE) &&
                    (STATE.user_outline_scale[1] == 0) &&
                    (STATE.user_outline_scale[2] == 0) &&
                    (STATE.user_outline_scale[3] == FIXED_ONE));

        if (!identity) scale_user_outline(_PS_ unitsPerEm);
    }
    else
    {
        STATE.user_outline_scale[0] = FIXED_ONE;
        STATE.user_outline_scale[1] = 0;
        STATE.user_outline_scale[2] = 0;
        STATE.user_outline_scale[3] = FIXED_ONE;
    }
    STATE.user_outline_scale_vanilla = ((STATE.user_outline_scale[0] > 0) &&
                                        (STATE.user_outline_scale[1] == 0) &&
                                        (STATE.user_outline_scale[2] == 0) &&
                                        (STATE.user_outline_scale[3] > 0));

    set_user_outline_bounds(_PS0_);

    if (need_to_set_advance)
    {
        if (STATE.user_outline_scale_vanilla)
        {
            STATE.user_outline->i_dx = (FS_SHORT)((STATE.user_outline->hi_x +
                                                   STATE.user_outline->lo_x +
                                                   0x8000) >> 16);
        }
        else
        {
            STATE.user_outline->i_dx = 0;
        }
        STATE.user_outline->i_dy = 0;

        STATE.user_outline->dx = STATE.user_outline->hi_x +
                                 STATE.user_outline->lo_x;
        STATE.user_outline->dy = 0;
    }
#else
    (void)outl;       /* unused */
    (void)doScaling;  /* unused */
    (void)unitsPerEm; /* unused */
#endif /* FS_EXTERNAL_OUTLINE */
#else
    (void)outl;       /* unused */
    (void)doScaling;  /* unused */
    (void)unitsPerEm; /* unused */
#endif /* FS_RENDER */
    return STATE.error = SUCCESS;
}
